<?php

namespace App\Classes;

use App\Exceptions\StripeException;
use App\Helper\Helper\NotificationHelper;
use App\Models\Payment;
use App\Repositories\Payment\Refund\RefundInterface;
use App\Services\LogService;
use Exception;
use Illuminate\Support\Facades\Log;
use GuzzleHttp\Exception\RequestException;

class Refund
{
    protected $refundRepo, $log;

    function __construct(RefundInterface $refundRepo, LogService $log)
    {
        $this->refundRepo = $refundRepo;
        $this->log = $log;

        // dd($log, $refundRepo);
    }

    public function singleRefund($paymentID, $reason, $amount, $refundStatus, $createdBy)
    {
        //$reason = !empty($reason['reason']) ? $reason['reason'] : 'requested_by_customer';

        $payment = Payment::select('id', 'price', 'intent_id', 'currency')->with(['currencyCountry:id,symbol', 'paymentLink:id,payment_gateway'])
            ->where('id', $paymentID)->first();
        //dd($payment);
        $logRefund = [
            'type' => 'refund.refunded',
            'log_type' => 'payment',
            'loggable_id' => $paymentID,
            'created_by' => $createdBy,
            'request' => response()->json([
                'payment_id' => $paymentID,
                'reason' => !empty($reason['reason']) ? $reason['reason'] : '',
                'detailed_reason' => !empty($reason['detailed_reason']) ? $reason['detailed_reason'] : '',
                'amount' => !empty($amount) ? $amount : $payment->price,
                'created_by' => $createdBy
            ])->content()
        ];

        try {
            $refundAmount = ($amount != 0 && $amount < $payment->price) ? $amount : $payment->price;
            $refund = $this->refundRepo->create($paymentID, $reason, $refundAmount * 100, $refundStatus);

            if (($amount != 0 && $amount < $payment->price)) {
                $logRefund['activity'] = "Successfully refunded " . $payment->currencyCountry?->symbol . $amount . " due to " . $reason['reason'] . ". It may take a few days for the money to reach the customer's bank account.";
            } else {
                $logRefund['activity'] = "Successfully refunded due to " . $reason['reason'] . " charge. It may take a few days for the money to reach the customer's bank account.";
            }

            $logRefund['response'] = response()->json($refund)->content();
            $logRefund['code'] = 200;

            // Log success or send notifications
            $this->log->log('payment', $logRefund);

            if (!empty($reason['detailed_reason'])) {
                $logDetailedReasonNote = [
                    'type' => 'noted',
                    'log_type' => 'payment',
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                    'request' => response()->json([
                        'payment_id' => $paymentID,
                        'reason' => !empty($reason['reason']) ? $reason['reason'] : '',
                        'detailed_reason' => !empty($reason['detailed_reason']) ? $reason['detailed_reason'] : '',
                        'amount' => !empty($amount) ? $amount : $payment->price,
                        'created_by' => $createdBy
                    ])->content(),
                    'activity' => $reason['detailed_reason'],
                    'response' => response()->json($refund)->content(),
                    'code' => 200,
                ];
                $this->log->log('payment', $logDetailedReasonNote);
            }

            NotificationHelper::notify('payment-refund', 'Payment-Refund: Amount of ' . $payment->currencyCountry?->symbol . $amount . ' ' . 'has been refunded', [
                'id' => $payment->id,
                'reason' => $reason,
                'message' => 'Payment refund successful',
            ], $createdBy);

            if ($amount < $payment->price) {
                $payment->status = 9;
            } else {
                $payment->status = 11;
            }
            $payment->save();

            return [
                "success" => true
            ];
        } catch (StripeException $e) {
            $logRefund['type'] = 'refund.failed';
            $logRefund['activity'] = $e->getMessage();
            $logRefund['response'] = response()->json(['error' => $e->getError()])->content();
            $logRefund['code'] = $e->getHttpStatusCode();

            if (strpos($e->getMessage(), "has already been refunded") !== false || strpos($e->getMessage(), "duplicate") !== false) {
                // $payment->status = 14;
                $logRefund['activity'] = $e->getMessage() . ' (From Stripe)';
                $payment->status = 11;
                $payment->save();
            }

            $this->log->log('payment', $logRefund);

            throw $e;
        } catch (Exception $e) {
            $message = explode("|", $e->getMessage());

            if (strpos($e->getMessage(), "has already been refunded") !== false || strpos($e->getMessage(), "duplicate") !== false) {
                // $payment->status = 14;
                $payment->status = 11;
                $payment->save();
            }

            $logRefund['type'] = 'refund.failed';
            $logRefund['activity'] = !empty($message[2]) ? $message[2] : $e->getMessage();
            $logRefund['response'] = response()->json([
                'error' => [
                    'message' => $e->getMessage(),
                    'code' => $e->getCode()
                ]
            ])->content();

            $logRefund['code'] = $e->getCode();
            $this->log->log('payment', $logRefund);

            throw $e;
        }
    }

    // for paypal
    public function singleRefundForPaypal($paymentID, $reason, $amount, $refundStatus, $createdBy)
    {
        //$reason = !empty($reason['reason']) ? $reason['reason'] : 'requested_by_customer';

        $payment = Payment::select('id', 'price', 'intent_id', 'currency')->with(['currencyCountry:id,symbol', 'paymentLink:id,payment_gateway'])
            ->where('id', $paymentID)->first();
        $logRefund = [
            'type' => 'refund.refunded',
            'log_type' => 'payment',
            'loggable_id' => $paymentID,
            'created_by' => $createdBy,
            'request' => response()->json([
                'payment_id' => $paymentID,
                'reason' => !empty($reason['reason']) ? $reason['reason'] : '',
                'detailed_reason' => !empty($reason['detailed_reason']) ? $reason['detailed_reason'] : '',
                'amount' => !empty($amount) ? $amount : $payment->price,
                'created_by' => $createdBy
            ])->content()
        ];

        try {
            $refundAmount = ($amount != 0 && $amount < $payment->price) ? $amount : $payment->price;
            $refund = $this->refundRepo->create($paymentID, $reason, $refundAmount, $refundStatus);

            if (($amount != 0 && $amount < $payment->price)) {
                $logRefund['activity'] = "Successfully refunded " . $payment->currencyCountry?->symbol . $amount . " due to " . $reason['reason'] . ". It may take a few days for the money to reach the customer's bank account.";
            } else {
                $logRefund['activity'] = "Successfully refunded due to " . $reason['reason'] . " charge. It may take a few days for the money to reach the customer's bank account.";
            }

            $logRefund['response'] = response()->json($refund)->content();
            $logRefund['code'] = 200;

            $this->log->log('payment', $logRefund);

            if (!empty($reason['detailed_reason'])) {
                $logDetailedReasonNote = [
                    'type' => 'noted',
                    'log_type' => 'payment',
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                    'request' => response()->json([
                        'payment_id' => $paymentID,
                        'reason' => !empty($reason['reason']) ? $reason['reason'] : '',
                        'detailed_reason' => !empty($reason['detailed_reason']) ? $reason['detailed_reason'] : '',
                        'amount' => !empty($amount) ? $amount : $payment->price,
                        'created_by' => $createdBy
                    ])->content(),
                    'activity' => $reason['detailed_reason'],
                    'response' => response()->json($refund)->content(),
                    'code' => 200,
                ];
                $this->log->log('payment', $logDetailedReasonNote);
            }

            NotificationHelper::notify('payment-refund', 'Payment-Refund: Amount of ' . $payment->currencyCountry->symbol . $amount . ' ' . 'has been refunded', [
                'id' => $payment->id,
                'reason' => $reason,
                'message' => 'Payment refund successful',
            ], $createdBy);

            if ($amount < $payment->price) {
                $payment->status = 9;
            } else {
                $payment->status = 11;
            }
            $payment->save();

            return [
                "success" => true
            ];
        }

        catch (RequestException $e) {
            if ($e->hasResponse()) {
                $errorResponse = json_decode($e->getResponse()->getBody()->getContents(), true);

                $logRefund['activity'] = '';

                if (!empty($errorResponse['details'][0]['description']) &&
                    strpos($errorResponse['details'][0]['description'], "The capture has already been fully refunded") !== false) {

                    $logRefund['activity'] = $errorResponse['details'][0]['description'];
                    // Update payment status
                    $payment->status = 11;
                    $payment->save();
                } else {
                    $logRefund['activity'] = !empty($errorResponse['message'])
                        ? $errorResponse['message']
                        : 'Refund failed w/o any message';
                }

                $logRefund['type'] = 'refund.failed';

                $logRefund['response'] = response()->json([
                    'error' => [
                        'message' => !empty($errorResponse['details'][0]['description'])
                            ? $errorResponse['details'][0]['description']
                            : 'No description available',
                        'code' => $e->getResponse()->getStatusCode(),
                    ]
                ])->content();

                $logRefund['code'] = $e->getResponse()->getStatusCode();

                $this->log->log('payment', $logRefund);
                throw $e;
            }

        }
    }
}
