<?php

namespace App\Repositories\WebBrief;

use App\Models\WebBrief;
use App\Services\BriefLogService;
use App\Helper\BriefHelper;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Spatie\Permission\Models\Role;

class WebBriefRepository implements WebBriefInterface
{

    private $webBrief, $briefLogService;

    public function __construct(WebBrief $webBrief, BriefLogService $briefLogService)
    {
        $this->webBrief = $webBrief;
        $this->briefLogService = $briefLogService;
    }

    public function getAllBriefs($request)
    {
        $briefs = $this->webBrief->select('id', 'encoded_id', 'brief_link_id', 'name', 'email', 'phone', 'business_name', 'created_at')
        ->with([
            'briefLink' => function ($link) {
                $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                    ->with([
                        'salesperson:id,first_name,last_name',
                        'customer:id,first_name,last_name,phone,email,company'
                    ]);
            },
        ])
        ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
            $parsedDate = null;
            if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
            }

            if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
            }
            switch ($request->dateSearchBy) {
                case '>':
                    if (!empty($request->date)) {
                        $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                        $query->whereDate('created_at', '>=', $date);
                    }
                    break;
                case '=':
                    if (!empty($request->date)) {
                        $query->whereDate('created_at', $parsedDate);
                    }
                    break;
                case '<>':
                    if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                        $startDate = Carbon::parse($request->dateBetween['start']);
                        $endDate = Carbon::parse($request->dateBetween['end']);

                        // Adjust time if provided
                        if (!empty($request->timeBetween['start'])) {
                            $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                        }

                        if (!empty($request->timeBetween['end'])) {
                            $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                        }

                        // Adjust timezone
                        if ($request->timezone == 'eastern') {
                            $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                            $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                        }

                        // Adjust timezone
                        if ($request->timezone == 'utc-5') {
                            $startDate->setTimezone('UTC')->addHours(5);
                            $endDate->setTimezone('UTC')->addHours(5);
                        }

                        // Apply date and time range filter
                        $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                    }
                    break;
                case '>=':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                            $query->where('created_at', '>=', $dateTime);
                        } else {
                            $query->whereDate('created_at', '>=', $parsedDate);
                        }
                    }
                    break;
                case '<':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                            $query->where('created_at', '<', $dateTime);
                        } else {
                            $query->whereDate('created_at', '<', $parsedDate);
                        }
                    }
                    break;
            }
        })
        ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->name . '%');
        })
        ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->where('email', $request->customerEmail);
        })
        ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
        })
        ->when(!empty($request->company), function ($query) use ($request) {
                $query->where('business_name', $request->company);
        })
        ->when(!empty($request->status), function ($query) use ($request) {
            $query->whereHas('briefLink', function ($q) use ($request) {
                $q->where('status', $request->status);
            });
        });

        $briefsCount = BriefHelper::getBriefsButtonCount('web_brief', [
            'title' => 'Web Brief',
            'count' => $briefs->count(),
            'active' => true,
            'route' => 'web.brief.list'
        ]);

        $briefs = $briefs->latest('id')->paginate(30)
        ->withQueryString();

        $mergedBriefs = $briefs->toArray();
        $mergedBriefs['buttonsData'] = $briefsCount->all();

        return $mergedBriefs;
    }

    public function getByEncodedID($id, $authenticated = false)
    {
        // Retrieve roles that have the 'BriefForm-Global' permission
        $rolesWithBriefGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'BriefForm-Global');
        })->pluck('name')->toArray();

        if ($authenticated && Auth::user()->hasAnyRole($rolesWithBriefGlobalPermission)) {
            $briefs = $this->webBrief->where('encoded_id', $id)->firstOrFail();
        } else {
            $briefs = $this->webBrief->excludeColumns(['name', 'email', 'phone'])->where('encoded_id', $id)->firstOrFail();
        }
        return $briefs;
    }

    public function arrayToString($array)
    {
        if (is_array($array)) {
            $string = implode(",", $array);
            return $string;
        }
    }

    public function getBrief($id){
        return $this->webBrief->with([
            'briefLink' => function ($link) {
                $link->with([
                        'salesperson',
                        'customer'
                    ]);
            },
        ])
        ->findOrFail($id);
    }

    public function createBrief($data)
    {
        try {
            $data = (object) $data;

            $webBrief = new $this->webBrief;

            $webBrief->brief_link_id = $data->brief_link_id;
            $webBrief->email = $data->email;
            $webBrief->name = $data->name;
            $webBrief->phone = $data->phone;
            $webBrief->business_name = $data->business_name;
            $webBrief->business_address = $data->business_address;
            $webBrief->business_phone = $data->business_phone;
            $webBrief->business_email = $data->business_email;
            $webBrief->social_media_handles = $data->social_media_handles;
            $webBrief->website = $data->website;
            $webBrief->web_hosting = $data->web_hosting;
            $webBrief->current_domain_address = $data->current_domain_address;
            $webBrief->organization_operations = $data->organization_operations;
            $webBrief->website_purpose = $data->website_purpose;
            $webBrief->portray = $this->arrayToString($data->portray);
            $webBrief->audience = $data->audience;
            $webBrief->pages = $data->pages;
            $webBrief->competitors = $data->competitors;
            $webBrief->website1_url = $data->website1_url;
            $webBrief->website1_comments = $data->website1_comments;
            $webBrief->website2_url = $data->website2_url;
            $webBrief->website2_comments = $data->website2_comments;
            $webBrief->website3_url = $data->website3_url;
            $webBrief->website3_comments = $data->website3_comments;
            $webBrief->website_features = $data->website_features;
            $webBrief->maintenance_team = $data->maintenance_team;
            $webBrief->website_content = $data->website_content;
            $webBrief->slogan = $data->slogan;
            $webBrief->have_logo = $data->have_logo;
            $webBrief->redesign_logo = $data->redesign_logo;
            $webBrief->website_fonts = $data->website_fonts;
            $webBrief->website_colors = $data->website_colors;
            $webBrief->anything_else = $data->anything_else;
            $webBrief->customer_id = $data->customer_id ?? Auth::user()->id;

            $webBrief->save();

            // $this->briefLogService->log([
            //     'brief_type' => 'web_brief',
            //     'brief_id' => $webBrief->id,
            //     'activity' => 'Web Brief has been created by ' . $webBrief->email . '.',
            //     'type' => 'web_brief.create',
            //     'created_by' => Auth::user()->id,
            // ]);

            $webBrief = $webBrief->refresh();

            return [
                'id' => $webBrief->id,
                'encoded_id' => $webBrief->encoded_id,
                'brief_link_id' => $webBrief->brief_link_id,
                'email' => $brief->email ?? '',
                'name' => $brief->name ?? '',
                'phone_number' => $brief->phone ?? '',
                'business_name' => $brief->business_name ?? '',
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }
}
