<?php

namespace App\Repositories\SmmBrief;

use App\Helper\BriefHelper;
use App\Models\SmmBrief;
use App\Services\BriefLogService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use App\Services\LogsService;
use Spatie\Permission\Models\Role;

class SmmBriefRepository implements SmmBriefInterface
{

    private $smmBrief, $briefLogService, $logsService;

    public function __construct(SmmBrief $smmBrief, BriefLogService $briefLogService)
    {
        $this->smmBrief = $smmBrief;
        $this->briefLogService = $briefLogService;
        $this->logsService = new LogsService('/logs/briefLogs');
    }

    public function getAllBriefs($request)
    {
        $briefs = $this->smmBrief->select('id', 'encoded_id', 'brief_link_id', 'name', 'email', 'phone', 'industry', 'business_name', 'created_at')
        ->with([
            'briefLink' => function ($link) {
                $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                    ->with([
                        'salesperson:id,first_name,last_name',
                        'customer:id,first_name,last_name,phone,email,company'
                    ]);
            },
        ])
        ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
            $parsedDate = null;
            if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
            }

            if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
            }
            switch ($request->dateSearchBy) {
                case '>':
                    if (!empty($request->date)) {
                        $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                        $query->whereDate('created_at', '>=', $date);
                    }
                    break;
                case '=':
                    if (!empty($request->date)) {
                        $query->whereDate('created_at', $parsedDate);
                    }
                    break;
                case '<>':
                    if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                        $startDate = Carbon::parse($request->dateBetween['start']);
                        $endDate = Carbon::parse($request->dateBetween['end']);

                        // Adjust time if provided
                        if (!empty($request->timeBetween['start'])) {
                            $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                        }

                        if (!empty($request->timeBetween['end'])) {
                            $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                        }

                        // Adjust timezone
                        if ($request->timezone == 'eastern') {
                            $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                            $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                        }

                        // Adjust timezone
                        if ($request->timezone == 'utc-5') {
                            $startDate->setTimezone('UTC')->addHours(5);
                            $endDate->setTimezone('UTC')->addHours(5);
                        }

                        // Apply date and time range filter
                        $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                    }
                    break;
                case '>=':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                            $query->where('created_at', '>=', $dateTime);
                        } else {
                            $query->whereDate('created_at', '>=', $parsedDate);
                        }
                    }
                    break;
                case '<':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                            $query->where('created_at', '<', $dateTime);
                        } else {
                            $query->whereDate('created_at', '<', $parsedDate);
                        }
                    }
                    break;
            }
        })
        ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->name . '%');
        })
        ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->where('email', $request->customerEmail);
        })
        ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
        })
        ->when(!empty($request->company), function ($query) use ($request) {
                $query->where('business_name', $request->company);
        })
        ->when(!empty($request->status), function ($query) use ($request) {
            $query->whereHas('briefLink', function ($q) use ($request) {
                $q->where('status', $request->status);
            });
        });

        $briefsCount = BriefHelper::getBriefsButtonCount('smm_brief', [
            'title' => 'SMM Brief',
            'count' => $briefs->count(),
            'active' => true,
            'route' => 'smm.brief.list'
        ]);

        $briefs = $briefs->latest('id')->paginate(30)
        ->withQueryString();

        $mergedBriefs = $briefs->toArray();
        $mergedBriefs['buttonsData'] = $briefsCount->all();

        return $mergedBriefs;
    }

    public function getByEncodedID($id, $authenticated = false)
    {
        // Retrieve roles that have the 'BriefForm-Global' permission
        $rolesWithBriefGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'BriefForm-Global');
        })->pluck('name')->toArray();

        if ($authenticated && Auth::user()->hasAnyRole($rolesWithBriefGlobalPermission)) {
            $briefs = $this->smmBrief->where('encoded_id', $id)->firstOrFail();
        } else {
            $briefs = $this->smmBrief->excludeColumns(['name', 'email', 'phone'])->where('encoded_id', $id)->firstOrFail();
        }
        return $briefs;
    }

    public function arrayToString($array)
    {
        if (is_array($array)) {
            $string = implode(",", $array);
            return $string;
        }
    }

    public function createBrief($data)
    {
        try {
            $data = (object) $data;

            $smmBrief = new $this->smmBrief;

            $smmBrief->brief_link_id = $data->brief_link_id;
            $smmBrief->email = $data->email;
            $smmBrief->name = $data->name;
            $smmBrief->phone = $data->phone;
            $smmBrief->business_name = $data->business_name;
            $smmBrief->industry = $data->industry;
            $smmBrief->business_email = $data->business_email;
            $smmBrief->services = $data->services;
            $smmBrief->business_goal = $data->business_goal;
            $smmBrief->slogan = $data->slogan;
            $smmBrief->company_address = $data->company_address;
            $smmBrief->unique_service = $data->unique_service;
            $smmBrief->audience = $data->audience;
            $smmBrief->website_link = $data->website_link;
            $smmBrief->website_hosting = $data->website_hosting;
            $smmBrief->competitors_website = $data->competitors_website;
            $smmBrief->facebook_link = $data->facebook_link;
            $smmBrief->instagram_link = $data->instagram_link;
            $smmBrief->pinterest_link = $data->pinterest_link;
            $smmBrief->social_media_competitors = $data->social_media_competitors;
            $smmBrief->customer_id = $data->customer_id ?? Auth::user()->id;

            $smmBrief->save();
            $smmBrief = $smmBrief->refresh();

            // $this->briefLogService->log([
            //     'brief_type' => 'logo_brief',
            //     'brief_id' => $brief->id,
            //     'activity' => 'Logo Brief has been created by ' . $brief->email . '.',
            //     'type' => 'logo_brief.create',
            //     'created_by' => Auth::user()->id,
            // ]);

            // $data = ['class_id' => $brief->id, 'class_type' => get_class($brief), 'activity' => 'Logo Brief has been created by ' . $brief->email . '.', 'type' => 'create', 'request' => '', 'response' => '', 'created_by' => Auth::user()->id, 'event' => null, 'pin' => 0];
            // $this->logsService->log($data);

            return [
                'id' => $smmBrief->id,
                'encoded_id' => $smmBrief->encoded_id,
                'brief_link_id' => $smmBrief->brief_link_id,
                'email' => $smmBrief->email ?? '',
                'name' => $smmBrief->name ?? '',
                'phone_number' => $smmBrief->phone ?? '',
                'business_name' => $smmBrief->business_name ?? '',
                'industry' => $smmBrief->industry ?? '',
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function getBrief($id){
        return $this->smmBrief->with([
            'briefLink' => function ($link) {
                $link->with([
                        'salesperson',
                        'customer'
                    ]);
            },
        ])
        ->findOrFail($id);
    }

}
