<?php

namespace App\Repositories\Product;

use App\Models\Category;
use App\Services\GenerateUniqueID;
use Illuminate\Support\Arr;
use Exception;

class ProductRepository implements ProductInterface {

  protected $productImagePath = 'backend/images/product/';

  private $generateUniqueID;

  function __construct(GenerateUniqueID $generateUniqueID)
  {
      $this->generateUniqueID = $generateUniqueID;
  }
  // Add or Update product
  public function addOrUpdate(...$args) {
    try {
        $id = $args[0]['id'] ?? null;
        $product = Category::findOrNew($id);
        $isNewProduct = $product->id == null;

        $product->name = $args[0]['name'];

        // Check if 'image' is present in the incoming data and is an instance of UploadedFile
        if (isset($args[0]['image']) && $args[0]['image'] instanceof \Illuminate\Http\UploadedFile) {
            $image = $args[0]['image'];
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $imagePath = $this->productImagePath . $imageName;
            $image->move($this->productImagePath, $imageName);

            $product->image = $imagePath;
        }
        if ($isNewProduct) {
            $prefix='prod';
            $length = 10;
            $uniqueProductId = $this->generateUniqueID->create($prefix, $length);
            $product->custom_id = $uniqueProductId;
            $product->created_by = $args[0]['created_by'];
        } else {
            $product->updated_by = $args[0]['updated_by'];
        }

        $product->save();

        $data['success'] = true;
        $data['id'] = $product->id;

        return $data;
    } catch(Exception $ex) {
        $data['success'] = false;
        $data['message'] = $ex->getMessage();

        return $data;
    }
}

  // Detail of product
  public function product($id) {
    return Category::find($id);
  }

  // list of products
  public function list($select="") {
    return Category::all();
  }

  // Archived product
  public function softDeleteProduct($id) {
    try {
        $product = Category::find($id);
      if (!empty($product)) {
          $product->status = 2;
          $product->save();
          $product->delete(); // Soft delete
          $data['id'] = $product->id;
          $data['success'] = true;
          $data['message'] = 'Product Archived Successfully';

          return $data;
      }
    } catch(Exception $ex) {
      $data['success'] = false;
      $data['message'] = $ex->getMessage();

      return $data;
    }
  }

  // Permanent Delete product
  public function forceDeleteProduct($id) {
    try {
        $product = Category::withTrashed()->find($id);
      if (!empty($product)) {
          $product->forceDelete(); // Permanent delete
          $data['success'] = true;
          $data['message'] = 'Product Deleted Successfully';

          return $data;
      }
    } catch(Exception $ex) {
      $data['success'] = false;
      $data['message'] = $ex->getMessage();

      return $data;
    }
  }

  // Restore product
  public function restoreProduct($id) {
    try {
        $product = Category::withTrashed()->find($id);
      if (!empty($product)) {
          $product->restore(); // Restore the soft-deleted record
          $product->status = 1;
          $product->save();
          $data['id'] = $product->id;
          $data['success'] = true;
          $data['message'] = 'Product Restored Successfully';

          return $data;
      }
    } catch(Exception $ex) {
      $data['success'] = false;
      $data['message'] = $ex->getMessage();

      return $data;
    }
  }

    // Add product metadata
    public function updateProductMetadata(...$args) {
        try {
          $id = $args[0]['id'] ?? null;
          $product = Category::find($id);

          if ($args[0]['metadata']) {
            $metadata = $args[0]['metadata'];
            
            $metadata = array_filter($metadata, function($item) {
                return $item['key'] !== null && $item['value'] !== null;
            });
    
            $metadata = Arr::pluck($metadata, 'value', 'key');
            $product->metadata = json_encode($metadata);
            $product->updated_by = $args[0]['updated_by'];
        }
         
          $product->save();

          $data['success'] = true;
          $data['id'] = $product->id;

          return $data;
      } catch(Exception $ex) {
          $data['success'] = false;
          $data['message'] = $ex->getMessage();

          return $data;
      }
    }

}
