<?php

namespace App\Repositories\Logs;

use Exception;
use Illuminate\Http\Request;

class LogRepository implements LogInterface
{
    private $storagePath;

    public function __construct($storagePath = null)
    {
        $formattedPath = $this->snakeToCamel($storagePath);
        $this->storagePath = storage_path($formattedPath);
    }

    /**
     * Covert String from Snack case to Camel Case
     * @param mixed $string
     * @return string
     */
    private function snakeToCamel($string)
    {
        $string = str_replace('_', ' ', $string);
        $string = ucwords($string);
        $string = str_replace(' ', '', $string);
        $string = lcfirst($string);

        return $string;
    }

    /**
     * Get the file path for a given class ID.
     *
     * @param string $id
     * @return string
     */
    private function getFilePath($id)
    {
        return $this->storagePath . '/' . $id . '.json';
    }

    /**
     * Load logs from a JSON file based on class ID.
     *
     * @param string $id
     * @return array
     */
    private function loadLog($id)
    {
        $filePath = $this->getFilePath($id);
        if (!file_exists($filePath)) {
            return [
                'class_type' => '',
                'class_id' => $id,
                'logs' => []
            ];
        }

        $existingContent = file_get_contents($filePath);
        return json_decode($existingContent, true) ?? ['logs' => []];
    }

    /**
     * Save the logs to a JSON file.
     *
     * @param string $id
     * @param array $log
     * @return void
     */
    private function saveLog($id, $log)
    {
        $filePath = $this->getFilePath($id);

        if (!is_dir($this->storagePath)) {
            mkdir($this->storagePath, 0755, true);
        }

        $logJson = json_encode($log, JSON_PRETTY_PRINT);
        file_put_contents($filePath, $logJson);
    }

    /**
     * Generate a unique ID for a log entry.
     *
     * @return string
     */
    private function generateLogId()
    {
        return uniqid();
    }

    /**
     * Format the incoming request for logging.
     *
     * @param \Illuminate\Http\Request|null $request
     * @return array|string
     */
    private function formatRequest($request)
    {
        if ($request instanceof Request) {
            try {
                $isJson = $request->isJson();
                $body = null;

                if ($isJson) {
                    $body = $request->json()->all();
                } elseif ($request->isMethod('GET')) {
                    $body = $request->query();
                } else {
                    $contentType = $request->header('Content-Type');

                    if (str_contains($contentType, 'application/x-www-form-urlencoded') || str_contains($contentType, 'multipart/form-data')) {
                        $body = $request->all();
                    } else {
                        $body = $request->getContent();
                    }
                }

                $formattedRequest = [
                    'method' => $request->method(),
                    'url' => $request->fullUrl(),
                    'path' => $request->path(),
                    // 'headers' => $request->headers->all(),  // Commented because these are not in use for now
                    'body' => $body,
                    // 'cookies' => $request->cookies->all(),  // Commented because these are not in use for now
                    // 'query' => $request->query(), // Commented because these are not in use for now
                    // 'user-Agent' => $request->header('User-Agent'),  // Commented because these are not in use for now
                ];

                return $formattedRequest;
            } catch (Exception $e) {
                return [
                    'error' => 'Failed to format request: ' . $e->getMessage(),
                ];
            }
        } elseif (is_string($request)) {
            return $request;
        }else {
            return 'Invalid request provided.';
        }
    }

    /**
     * Log a new entry into the logs.
     *
     * @param array $data
     * @return array
     * @throws Exception
     */
    public function log($data)
    {
        try {
            $logs = $this->loadLog($data['class_id']);

            if (empty($logs['class_type'])) {
                $logs['class_type'] = $data['class_type'];
            }

            $newLogEntry = [
                'id' => $this->generateLogId(),
                'activity' => $data['activity'],
                'type' => $data['type'],
                'request' => $this->formatRequest($data['request'] ?? null),
                'response' => $data['response'] ?? '',
                'created_by' => $data['created_by'],
                'code' => $data['code'] ?? null,
                'event' => $data['event'] ?? 0,
                'pin' => $data['pin'] ?? 0,
                'created_at' => date('Y-m-d H:i:s')
            ];

            $logs['logs'][] = $newLogEntry;

            $this->saveLog($data['class_id'], $logs);

            return $logs;
        } catch (Exception $e) {
            throw new Exception("Failed to log: " . $e->getMessage());
        }
    }

    /**
     * Find log index by its ID.
     *
     * @param string $logId
     * @param array $logs
     * @return int|null
     */
    private function findLogIndexById($logId, $logs)
    {
        foreach ($logs['logs'] as $index => $log) {
            if ($log['id'] === $logId) {
                return $index;
            }
        }
        return null;
    }

    /**
     * Update an existing log entry by its ID.
     *
     * @param string $logId
     * @param string $classID
     * @param array $data
     * @return array
     * @throws Exception
     */
    public function updateLog($logId, $classID, $data)
    {
        try {
            $logs = $this->loadLog($classID);
            $logIndex = $this->findLogIndexById($logId, $logs);

            if ($logIndex !== null) {
                $logs['logs'][$logIndex] = array_merge($logs['logs'][$logIndex], $data);
                $this->saveLog($classID, $logs);
            } else {
                throw new Exception("Log with ID $logId not found.");
            }

            return $logs['logs'][$logIndex];
        } catch (Exception $e) {
            throw new Exception("Failed to update log: " . $e->getMessage());
        }
    }

    /**
     * Pin or unpin a log entry by its ID. Only one log entry per class can be pinned.
     *
     * @param string $logId
     * @param string $classID
     * @param int $pin
     * @return array
     * @throws Exception
     */
    public function pinUnpinLog($logId, $classID, $pin)
    {
        try {
            $logs = $this->loadLog($classID);
            $logIndex = $this->findLogIndexById($logId, $logs);

            if ($pin == 1) {
                foreach ($logs['logs'] as &$log) {
                    $log['pin'] = 0;
                }
            }

            if ($logIndex !== null) {
                $logs['logs'][$logIndex]['pin'] = $pin;
            } else {
                throw new Exception("Log with ID $logId not found.");
            }

            $this->saveLog($classID, $logs);

            return $logs['logs'][$logIndex];
        } catch (Exception $e) {
            throw new Exception("Failed to pin/unpin log: " . $e->getMessage());
        }
    }

    /**
     * Retrieve a specific log entry by its ID.
     *
     * @param string $logId
     * @param string $classID
     * @return array
     * @throws Exception
     */
    public function getLog($logId, $classID)
    {
        try {
            $logs = $this->loadLog($classID);
            $logIndex = $this->findLogIndexById($logId, $logs);

            if ($logIndex !== null) {
                return $logs['logs'][$logIndex];
            } else {
                throw new Exception("Log with ID $logId not found.");
            }
        } catch (Exception $e) {
            throw new Exception("Failed to retrieve log: " . $e->getMessage());
        }
    }

    /**
     * Delete a specific log entry by its ID.
     *
     * @param string $logId
     * @param string $classID
     * @return bool
     * @throws Exception
     */
    public function deleteLog($logId, $classID)
    {
        try {
            $logs = $this->loadLog($classID);
            $logIndex = $this->findLogIndexById($logId, $logs);

            if ($logIndex !== null) {
                array_splice($logs['logs'], $logIndex, 1);
                $this->saveLog($classID, $logs);
                return true;
            } else {
                throw new Exception("Log with ID $logId not found.");
            }
        } catch (Exception $e) {
            throw new Exception("Failed to delete log: " . $e->getMessage());
        }
    }

    /**
     * Get all logs for a specific class ID.
     *
     * @param string $classID
     * @return array
     * @throws Exception
     */
    public function getAllLogs($classID)
    {
        try {
            $logs = $this->loadLog($classID);

            usort($logs['logs'], function($a, $b) {
                return strtotime($b['created_at']) - strtotime($a['created_at']);
            });

            return [
                'class_id' => $logs['class_id'],
                'class_type' => $logs['class_type'],
                'logs' => $logs['logs']
            ];
        } catch (Exception $e) {
            throw new Exception("Failed to retrieve all logs: " . $e->getMessage());
        }
    }
}
