<?php

namespace App\Models;

use App\Helper\Helper\StripeHelper;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class Payment extends Model
{
    use HasFactory;
    protected $fillable = ['id', 'customer_id', 'payment_link_id', 'price', 'discount', 'currency', 'ip', 'intent_id', 'charge_id', 'charge', 'refund', 'intent', 'paymentMethod', 'last_four', 'bin', 'coupon_id', 'converted_amount', 'comment', 'gateway_message', 'device', 'location', 'status', 'created_at'];

    public function paymentLogs()
    {
        return $this->morphMany(PaymentLog::class, 'loggable');
    }

    public function currencyCountry()
    {
        return $this->hasOne(CountryCurrencies::class, 'id', 'currency');
    }

    public function customer()
    {
        return $this->hasOne(User::class, 'id', 'customer_id');
    }

    public function link()
    {
        return $this->hasOne(PaymentLink::class, 'id', 'payment_link_id');
    }

    public function invoice()
    {
        return $this->belongsTo(Invoice::class, 'id', 'payment_id');
    }

    public function paymentLink()
    {
        return $this->belongsTo(PaymentLink::class, 'payment_link_id');
    }

    public function relatedPayments()
    {
        return $this->hasMany(RelatedPayments::class, 'payment_id');
    }

    public static function queryStatusCount($mainQuery, $statuses, $user = "", $role = "admin", $request = null)
    {
        $filteredRequest = count($request->except(['replace', 'preserveState', 'status_all', 'order', 'key', 'page'])) > 0;

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;
        $paymentSums = $mainQuery
        ->whereHas('customer', function ($query) {
            $query->whereNull('deleted_at');
        })
        ->where('status', '<>', 13)
        ->select('status', DB::raw('COUNT(DISTINCT(payment_link_id)) as total'))
        ->groupBy('status');

        if ($user !== "" && !$authUser->can('Payment-Global')) {
            $paymentSums = $paymentSums->whereHas('link', function ($query) use ($user) {
                $query->where('created_by', $user);
            });
        }

        if ($authUser->can('Payment-Global') && $authUserRole != "Super Admin") {
            $superAdminIds = User::whereHas('roles', function ($query) {
                $query->where('name', 'Super Admin');
            })->pluck('id');

            $paymentSums = $paymentSums->whereHas('link', function ($query) use ($superAdminIds) {
                $query->whereNotIn('created_by', $superAdminIds);
            });
        }

        if (!$filteredRequest) {
            $paymentSums = $paymentSums->whereHas('link', function ($query) {
                $query->whereMonth('created_at', Carbon::now()->month);
            });
        }

        $paymentSums = $paymentSums->get()->map(function ($column) use ($statuses) {
            $column['status'] = $statuses[$column['status']];
            return $column;
        });

        $paymentSums = collect(Arr::pluck($paymentSums->toArray(), 'total', 'status'));
        $paymentSums->put('all', $paymentSums->sum());

        return $paymentSums;
    }

    public static function statusCount($statuses, $user = "", $role = "admin")
    {
        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;
        $paymentSums = (new static)::query()
            ->whereHas('customer', function ($query) {
                $query->where('deleted_at', null);
            })
            ->where('status', '<>', 13)
            ->select('status', DB::raw('COUNT(status) as total'))
            ->groupBy('status');

        if ($user !== "" && !$authUser->can('Payment-Global')) {
            $paymentSums = $paymentSums->whereHas('link', function ($query) use ($user) {
                $query->where('created_by', $user);
            });
        }

        if ($authUser->can('Payment-Global') && $authUserRole != "Super Admin") {
            $superAdminIds = User::whereHas('roles', function ($query) {
                $query->where('name', 'Super Admin');
            })->pluck('id');

            $paymentSums = $paymentSums->whereHas('link', function ($query) use ($superAdminIds) {
                $query->whereNotIn('created_by', $superAdminIds);
            });
        }

        // if($role == 'Brand Manager') {
        //     $adminUserIds = User::whereHas('roles.permissions', function ($query) {
        //         $query->where('name', 'Payment-Global');
        //     })->pluck('id');
        //     // $adminUsers = User::select('id','email')->role('Admin')->get();

        //     // // Get IDs of admin users
        //     // $adminUserIds = $adminUsers->pluck('id');

        //     $paymentSums = $paymentSums->whereHas('link', function ($query) use ($adminUserIds) {
        //         $query->whereNotIn('created_by', $adminUserIds);
        //     });
        // }

        $paymentSums = $paymentSums->get()->map(function ($column) use ($statuses) {
            $column['status'] = $statuses[$column['status']];

            return $column;
        });

        $paymentSums = collect(Arr::pluck($paymentSums->toArray(), 'total', 'status'));
        $paymentSums->put('all', $paymentSums->sum());

        return $paymentSums;
    }


    public static function userStatusCount($statuses, $request)
    {
        $paymentSums = (new static)::query()
            ->whereHas('link', function ($link) use ($request) {
                $link->where('created_by', $request->id);
            })->select('status', DB::raw('COUNT(status) as total'))
            ->groupBy('status')
            ->get()->map(function ($column) use ($statuses) {
                $column['status'] = $statuses[$column['status']];

                return $column;
            });

        $paymentSums = collect(Arr::pluck($paymentSums->toArray(), 'total', 'status'));
        $paymentSums->put('all', $paymentSums->sum());

        return $paymentSums;
    }


    public static function sumCount($statuses, $dateFrom = '', $dateTo = '', $userId = '')
    {
        $paymentSums = (new static)::query()
            ->select('status', DB::raw('SUM(price) as total'))
            ->groupBy('status');

        if ($dateFrom != '' && $dateTo != '') {
            $paymentSums = $paymentSums->whereDate('created_at', '>=', $dateFrom);
            $paymentSums = $paymentSums->whereDate('created_at', '<=', $dateTo);
        }

        if (!empty($userId)) {
            $paymentSums = $paymentSums->where('customer_id', $userId);
        }

        $paymentSums = $paymentSums->get()->map(function ($column) use ($statuses) {
            $column['status'] = $statuses[$column['status']];

            return $column;
        });

        $paymentSums = collect(Arr::pluck($paymentSums->toArray(), 'total', 'status'));
        $paymentSums->put('all', $paymentSums->sum());

        return $paymentSums;
    }

    public static function lastPayments(string $status = "", $dateFrom = "", $dateTo = "", int $limit = 5)
    {
        $statuses = StripeHelper::getStatuses([1, 2, 3, 11, 5]);

        $statusWisePayments = (new static)::query()
            ->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'status', 'created_at')
            ->with('customer:id,email', 'currencyCountry:id,code,symbol')
            ->whereHas('customer', function ($query) {
                $query->where('deleted_at', null);
            })
            ->when($status != '', function ($query) use (&$status, &$statuses) {
                $query->whereStatus($statuses[$status]);
            })
            ->when($dateFrom != "" && $dateTo != "", function ($query) use (&$dateFrom, &$dateTo) {
                $query->whereDate('created_at', '>=', $dateFrom);
                $query->whereDate('created_at', '<=', $dateTo);
            })
            ->take($limit)
            ->latest('id');

        return $statusWisePayments;
    }

    public static function sumCountbyId($statuses, $dateFrom = '', $dateTo = '', $userId = '')
    {
        $superAdminIds = User::whereHas('roles', function ($query) {
            $query->where('name', 'Super Admin');
        })->pluck('id');

        $paymentSums = (new static)::query()
            ->select('status', DB::raw('SUM(price) as total'))
            ->groupBy('status');

        if ($dateFrom != '' && $dateTo != '') {
            $paymentSums = $paymentSums->whereDate('created_at', '>=', $dateFrom);
            $paymentSums = $paymentSums->whereDate('created_at', '<=', $dateTo);
        }

        if (!empty($userId)) {
            $paymentSums = $paymentSums->whereHas('link', function ($query) use ($userId) {
                $query->where('created_by', $userId);
            });
        } else {
            $paymentSums = $paymentSums->whereHas('link', function ($query) use ($superAdminIds) {
                $query->whereNotIn('created_by', $superAdminIds);
            });
        }

        $paymentSums = $paymentSums->get()->map(function ($column) use ($statuses) {
            $column['status'] = $statuses[$column['status']];

            return $column;
        });

        $paymentSums = collect(Arr::pluck($paymentSums->toArray(), 'total', 'status'));
        $paymentSums->put('all', $paymentSums->sum());

        return $paymentSums;
    }

    public static function baseQuery($status="")
    {
        $mainQuery = self::query()
            ->select('id', 'customer_id', 'intent_id', 'payment_link_id', 'price', 'discount', 'currency', 'ip', 'comment', 'gateway_message', 'paymentMethod', 'last_four', 'status', 'created_at')
            ->with([
                'link' => function ($link) {
                    $link->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'discount_type', 'discount', 'original_price', 'item_description', 'currency', 'payment_gateway', 'sale_type', 'team', 'created_by')
                        ->with([
                            'currencyCountry:id,code,symbol',
                            'creator:id,first_name,last_name',
                            'gateway:id,name',
                            'teamWise:id,name,color',
                        ]);
                },
                'customer:id,email,first_name,last_name',
                // Checked there wasn't any need - uncomment if needed
                // 'paymentLogs' => function ($logs) {
                //     $logs->orderBy('created_at', 'DESC')
                //          ->orderBy('id', 'DESC');
                // }
            ])
            ->whereHas('customer', function ($query) {
                $query->whereNull('deleted_at');
            })
            ->where('status', '<>', 13);

            if($status != "") {
                $mainQuery = $mainQuery->where('status', '<>', $status);
            }

            return $mainQuery;
    }

    public static function getCategorizedData($query, array $statuses, $filteredRequest = false)
    {
        $baseQuery = $query->whereHas('link');
        $baseQuery->select(DB::raw('COUNT(id) as total'), DB::raw('SUM(price) as total_price'), 'status');

        if (!$filteredRequest) {
            $baseQuery->whereMonth('created_at', Carbon::now()->month);
        }

        return $baseQuery
            ->whereIn('status', $statuses)
            ->groupBy('status')
            ->get();
    }

    public static function getUpSellAndFreshSell($query, $filteredRequest)
    {
        $baseQuery = $query->where('status', 1);
        $linkIDs = $baseQuery->pluck('payment_link_id')->toArray();
        $paymentLink = new PaymentLink();

        if (!$filteredRequest) {
            $baseQuery->whereMonth('created_at', Carbon::now()->month);
        }

        $monthlyData = $baseQuery->get()
            ->groupBy('link.sale_type')
            ->map(function ($group) {
                $firstLink = $group->first();

                return (object) [
                    'sales_type' => $firstLink->paymentLink->sale_type,
                    'total_records' => $group->unique('payment_link_id')->count(),
                    'total_price' => $group->sum('price'),
                ];
            });


            /*
            * Should be removed
            */
        // $monthlyData = (clone $query)
        //     ->where('payments.status', 1)
        //     ->whereMonth('payments.created_at', Carbon::now()->month)
        //     ->selectRaw('
        //         (SELECT sale_type FROM payment_links WHERE payment_links.id = payments.payment_link_id) as sales_type,
        //         COUNT(DISTINCT payments.payment_link_id) as total_records,
        //         SUM(payments.price) as total_price
        //     ')
        //     ->groupBy('sales_type')
        //     ->get();




        // $completeData = (clone $baseQuery)
        //     ->selectRaw('payment_links.sale_type as sales_type, COUNT(DISTINCT(payments.payment_link_id)) as total_records, SUM(payments.price) as total_price')
        //     ->groupBy('payment_links.sale_type')
        //     ->get();

        return [
            'monthly' => $monthlyData,
            'complete' => $paymentLink->getUpSellAndFreshSellData(null, null, $linkIDs)
        ];
    }

    public static function getOtherInformation($query, $statuses, $request = null)
    {
        $filteredRequest = count($request->except(['replace', 'preserveState', 'status_all', 'order', 'key', 'page'])) > 0;

        $paymentQuery = $query;

        $otherInformation = [
            'fresh_sales' => [
                'status' => 'Fresh Sales',
                'color' => '#299CDB',
                'total' => 0,
                'completeTotal' => 0,
                'totalPrice' => 0,
                'completePrice' => 0,
            ],
            'upsell' => [
                'status' => 'Up Sell',
                'color' => '#ff59f2',
                'total' => 0,
                'completeTotal' => 0,
                'totalPrice' => 0,
                'completePrice' => 0,
            ],
            'failed' => [
                'status' => 'Failed',
                'color' => "#C1163F",
                'totalPrice' => 0,
            ],
            'refunded' => [
                'status' => 'Refunded',
                'color' => "#ff6552",
                'totalPrice' => 0,
            ],
            'total' => [
                'status' => 'Total',
                'color' => "#6abd42",
                'totalPrice' => 0,
            ]
        ];
        $categorizedData = self::getCategorizedData(clone $paymentQuery, [1, 2, 11], $filteredRequest);
        $salesRecords = self::getUpSellAndFreshSell(clone $paymentQuery, $filteredRequest);

        foreach ($salesRecords as $key => $recordType){
            foreach ($recordType as $record) {
                if (isset($otherInformation[$record->sales_type])) {
                    $otherInformation[$record->sales_type][$key == 'complete' ? $key . 'Total' : 'total'] += $record->total_records;
                    $otherInformation[$record->sales_type][$key == 'complete' ? $key . 'Price' : 'totalPrice'] += $record->total_price;
                }
            }
        }

        foreach ($categorizedData as $data) {
            $status = $data->status == 1 ? 'total' : ($statuses[$data->status] ?? 'unknown');
            if (isset($otherInformation[$status])) {
                $otherInformation[$status]['totalPrice'] = $data->total_price;
            }
        }

        $orderedKeys = ['total', 'fresh_sales', 'upsell', 'failed', 'refunded'];
        $orderedInformation = [];

        foreach ($orderedKeys as $key) {
            if (isset($otherInformation[$key])) {
                $orderedInformation[$key] = $otherInformation[$key];
            }
        }

        return $orderedInformation;
    }

    static function refundCount($startDate = "", $endDate = "", $userIDs = "", $userInclusion = false) {
        $refunds = self::select(DB::raw('COUNT(status) as total'), DB::raw('DATE(created_at) as date'));
        $refunds = $refunds->where('status', 11);

        if($userIDs != "") {
            $refunds = $refunds->whereHas('link', function ($query) use ($userIDs, $userInclusion) {
                if(gettype($userIDs) == 'array' || gettype($userIDs) == 'object') {
                    if($userInclusion) {
                        $query->whereIn('created_by', $userIDs);
                    } else {
                        $query->whereNotIn('created_by', $userIDs);
                    }
                } else {
                    if($userInclusion) {
                        $query->where('created_by', $userIDs);
                    } else {
                        $query->where('created_by', '<>', $userIDs);
                    }
                }
            });
        }

        if($startDate != "") {
            $refunds = $refunds->whereDate('created_at', '>=', $startDate);
        }
        if($endDate != "") {
            $refunds = $refunds->whereDate('created_at', '<=', $endDate);
        }
        $refunds = $refunds->groupBy('date')->get();

        return $refunds;
    }

    static function refundData() {

    }

    static function getSaleTypeCountFilter($saleType, $startDate = "", $endDate="", $userIDs = "", $userInclusion = false, $status = 1) {
        $saleTypeCount = self::where('status', $status);

        if($startDate != "") {
            $saleTypeCount = $saleTypeCount->whereDate('created_at', '>=', $startDate);
        }

        if($endDate != "") {
            $saleTypeCount = $saleTypeCount->whereDate('created_at', '<=', $endDate);
        }

        $saleTypeCount = $saleTypeCount->whereHas('link', function ($query) use ($saleType, $userIDs, $userInclusion) {
                if($userIDs != "") {
                    if(gettype($userIDs) == 'array' || gettype($userIDs) == 'object') {
                        $inclusion = $userInclusion ? 'whereIn' : 'whereNotIn';
                        $query->{$inclusion}('created_by', $userIDs);
                    } else {
                        if($userInclusion) {
                            $query->where('created_by', $userIDs);
                        } else {
                            $query->where('created_by', '<>', $userIDs);
                        }
                    }
                }

                $query->where('sale_type', $saleType);
            })
            ->with(['link' => function ($query) use ($saleType) {
                $query->selectRaw('sale_type');
            }])
            ->selectRaw('SUM(price) as count')
            ->first();

        return $saleTypeCount->count ?? 0;
    }

    // Scope Filters
    // Amount filter
    public function scopeFilterByAmount($query, $request, $value, $operator)
    {
        if (!empty($request->{$value}) && !empty($request->{$operator})) {
            $query->where('price', $request->{$operator}, $request->{$value});
        }
        return $query;
    }

    // Currency Filter
    public function scopeFilterByCurrency($query, $request)
    {
        if (!empty($request->currency)) {
            $query->whereHas('currencyCountry', function ($query) use ($request) {
                $query->where('code', $request->currency);
            });
        }
        return $query;
    }

    // Card Filter
    public function scopeFilterByCard($query, $request)
    {
        if (!empty($request->card)) {
            $query->whereRaw("JSON_EXTRACT(paymentMethod, '$.card.brand') LIKE ?", ['%' . $request->card . '%']);
            $query->orWhereRaw("JSON_EXTRACT(paymentMethod, '$.card.display_brand') LIKE ?", ['%' . $request->card . '%']);
        }
        return $query;
    }

    // Card Last Four Digits Filter
    public function scopeFilterByLastFour($query, $request)
    {
        if (!empty($request->lastFour)) {
            $query->where('last_four', $request->lastFour);
        }
        return $query;
    }

    // Customer Data Filter
    function scopeFilterByCustomerData($query, $request) {
        $query->whereHas('customer', function ($userQuery) use ($request) {
            if(!empty($request->customerId)) {
                $userQuery->where('stripe_customer_id', $request->customerId);
            }

            if(!empty($request->customerEmail)) {
                $userQuery->where('email', $request->customerEmail);
            }
        });
        return $query;
    }

    // Payment Link Filter
    function scopeFilterByPaymentLink($query, $request) {
        $query->whereHas('link', function ($linkQuery) use ($request) {
            if(!empty($request->saleType)) {
                $linkQuery->where('sale_type', $request->saleType);
            }

            if(!empty($request->team)) {
                $linkQuery->where('team', $request->team);
            }

            if(!empty($request->gateways)) {
                $linkQuery->where('payment_gateway', $request->gateways);
            }
        });
        return $query;
    }
    /*
    public function scopeFilterByField($query, $request, $field, $operator, $value)
    {
        if (!empty($request->{$value}) && !empty($request->{$operator})) {
            $query->where($field, $request->{$operator}, $request->{$value});
        }
        return $query;
    }
    */
}
