<?php

namespace App\Jobs;

use App\Helper\Helper\NotificationHelper;
use App\Models\Payment;
use App\Repositories\Payment\Log\LogRepository;
use App\Services\LogService;
use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Stripe\StripeClient;

class CheckPaymentRefunded implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $paymentID, $intentID, $price, $secretKey, $createdBy, $refundRepo, $log;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct($paymentID, $intentID, $price, $secretKey, $createdBy)
    {
        $this->paymentID = $paymentID;
        $this->intentID = $intentID;
        $this->price = $price;
        $this->secretKey = $secretKey;
        $this->createdBy = $createdBy;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        try {
            $stripe = new StripeClient($this->secretKey);
            $refund = $stripe->refunds->all(['payment_intent' => $this->intentID]);

            $directory = 'intents'; 

            if (!Storage::exists($directory)) {
                Storage::makeDirectory($directory, 0755, true);
            }

            $filePath = "$directory/".$this->intentID.".json"; // Define the path to your text file

            Storage::put($filePath, response()->json($refund)->content());

            if(!empty($refund['data'])) {

                foreach($refund['data'] as $refundRecord) {
                    $amount = !empty($refundRecord->amount) && $refundRecord->amount > 0 ? $refundRecord->amount / 100 : 0;

                    $logRefund = [
                        'type' => 'refund.refunded',
                        'activity' => $refundRecord->payment_intent.' has already been refunded',
                        'log_type' => 'payment',
                        'loggable_id' => $this->paymentID,
                        'created_by' => $this->createdBy,
                        'request' => response()->json([
                            'payment_id' => $this->paymentID,
                            'reason' => $refundRecord->reason,
                            'detailed_reason' => '',
                            'amount' => $amount,
                            'created_by' => $this->createdBy
                        ])->content()
                    ];

                    $log = new LogService(app(LogRepository::class));
                    $log->log('payment', $logRefund);

                    Payment::where('id', $this->paymentID)->update([
                        'refund' => response()->json($refundRecord)->content(),
                        'status' => 11,
                        'updated_at' => Carbon::now()
                    ]);

                    $paymentIntent = $refundRecord->payment_intent;
                    Log::info('Refunded intent: '. $paymentIntent);
                }
            }
        } catch(Exception $e) {
            Log::error($e->getMessage());
            $this->fail($e);
        }
    }
}
