<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\LeadRequest;
use App\Models\Lead;
use App\Models\User;
use App\Services\LeadService;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Helper\ActivityLogHelper;
use Illuminate\Support\Facades\Validator;
use App\Services\MetadataService;

class LeadController extends Controller
{
    private $leadService, $leadObj;


    public function __construct(LeadService $leadService, Lead $lead)
    {
        $this->leadService = $leadService;
        $this->leadObj = new $lead;
    }

    public function index(Request $request)
    {
        $columns = [
            ['key' => 'name', 'component' => 'NameColumn', 'text' => 'Name', 'visible' => true, 'fixed' => true],
            ['key' => 'email', 'component' => 'EmailColumn', 'text' => 'Email', 'visible' => true],
            ['key' => 'phone', 'component' => 'PhoneColumn', 'text' => 'Phone', 'visible' => true],
            ['key' => 'status', 'component' => 'StatusColumn', 'text' => 'Lead Status', 'visible' => true],
            ['key' => 'source', 'component' => 'SourceColumn', 'text' => 'Source', 'visible' => true],
            ['key' => 'type', 'component' => 'TypeColumn', 'text' => 'Intent', 'visible' => true],
            ['key' => 'contact_medium', 'component' => 'ContactMediumColumn', 'text' => 'Contact Medium', 'visible' => true],
            ['key' => 'probability', 'component' => 'ProbabilityColumn', 'text' => 'Probability (%)', 'visible' => true],
            ['key' => 'rating', 'component' => 'RatingColumn', 'text' => 'Rating', 'visible' => true],
            ['key' => 'salesperson', 'component' => 'SalesPersonColumn', 'text' => 'SalesPerson', 'visible' => true],
            ['key' => 'date', 'component' => 'DateColumn', 'text' => 'Date', 'visible' => true],
        ];

        $leads = $this->leadService->getLeads($request);

        $tabs = [
            [
                'name' => 'All Leads',
                'route' => 'admin.lead.list',
                'statusKey' => '',
                'count' => $this->leadService->getLeadCount(''),
                'isActive' => $request->status == 'all' || empty($request->status),
                ]
            ];

            foreach ($this->leadObj->getStatusValues() as $lead) {
                $tabs[] = [
                'name' => $lead,
                'route' => 'admin.lead.list',
                'statusKey' => $lead,
                'count' => $this->leadService->getLeadCount($lead),
                'isActive' => $lead == $request->status,
            ];
        }

        $page = [
            'title' => 'Leads',
        ];

        $options = [
            'typeOptions' => $this->leadObj->getTypeValues(),
            'sourceOptions' => $this->leadObj->getSourceValues(),
            'statusOptions' => $this->leadObj->getStatusValues(),
            'ratingOptions' => $this->leadObj->getRatingValues(),
            'contactMediumOptions' => $this->leadObj->getContactMediumValues(),
        ];

        return Inertia::render('Admin/Lead/List', [
            'columns' => $columns,
            'page' => $page,
            'leads' => $leads,
            'options' => $options,
            'tabs' => $tabs,
        ]);
    }

    public function detail(Request $request)
    {
        $lead = $this->leadService->getLead($request->id);

        $page = [
            'title' => 'Lead Detail',
        ];

        return Inertia::render('Admin/Lead/Detail', [
            'lead' => $lead,
            'page' => $page
        ]);
    }

    public function add()
    {
        $salespersons = User::whereHas('roles', function ($query) {
            $query->where('name', 'Salesperson');
        })->get();

        $page = [
            'title' => 'Add Lead',
            'method' => 'add',
        ];

        $options = [
            'typeValues' => $this->leadObj->getTypeValues(),
            'sourceValues' => $this->leadObj->getSourceValues(),
            'statusValues' => $this->leadObj->getStatusValues(),
            'ratingValues' => $this->leadObj->getRatingValues(),
            'contactMediumValues' => $this->leadObj->getContactMediumValues(),
        ];

        return Inertia::render('Admin/Lead/AddEdit', [
            'page' => $page,
            'salespersons' => $salespersons,
            'options' => $options,
        ]);
    }

    public function create(LeadRequest $request)
    {
        $lead = $this->leadService->create($request->all());

        // Log the activity before logging out
        // ActivityLogHelper::staticLogActivity(
        //     'lead-created',
        //     'New Lead added successfully'
        // );

        // return redirect()->route('admin.lead.list');
        return response()->json(['success' => true, 'message' => 'Lead added successfully']);
    }

    public function edit(Request $request)
    {
        $lead = $this->leadService->getLead($request->id);

        $salespersons = User::whereHas('roles', function ($query) {
            $query->where('name', 'Salesperson');
        })->get();

        $page = [
            'title' => 'Edit Lead',
            'method' => 'edit',
        ];

        $options = [
            'typeValues' => $this->leadObj->getTypeValues(),
            'sourceValues' => $this->leadObj->getSourceValues(),
            'statusValues' => $this->leadObj->getStatusValues(),
            'ratingValues' => $this->leadObj->getRatingValues(),
            'contactMediumValues' => $this->leadObj->getContactMediumValues(),
        ];

        return Inertia::render('Admin/Lead/AddEdit', [
            'lead' => $lead,
            'page' => $page,
            'salespersons' => $salespersons,
            'options' => $options,
        ]);
    }

    public function update(LeadRequest $request)
    {
        $lead = $this->leadService->update($request->all());

        // Log the activity before logging out
        // ActivityLogHelper::staticLogActivity(
        //     'lead-updated',
        //     'Lead has been updated successfully'
        // );

        // return redirect()->route('admin.lead.list');
        return response()->json(['success' => true, 'message' => 'Lead updated successfully']);
    }

    public function delete(Request $request)
    {
        $lead = $this->leadService->delete($request->id);

        return redirect()->route('admin.lead.list');
        // return response()->json(['success' => true, 'message' => 'Lead deleted successfully']);
    }

    public function updateValue(Request $request){

        $rules = [
            'field' =>'required|string',
            'value' =>'required|string'
        ];

        $message = [
            'field.required' => 'Field is required',
            'value.required' => 'Value is required',
        ];

        $validator = Validator::make($request->all(), $rules, $message);

        if($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->first()]);
        }

        $leadStatuses = [
            'new' => 10,
            'contacted' => 30,
            'nurturing' => 50,
            'qualified' => 80,
            'unqualified' => 80,
        ];

        $column = $request->field == 'type' || $request->field == 'source' || $request->field == 'status' ? "lead_$request->field" : $request->field;

        $lead = Lead::find($request->id);
        $lead->{$column} = $request->value;
        $lead->probability = $leadStatuses[$lead->lead_status];
        $lead->save();

        return redirect()->route('admin.lead.list', ["status" => $request->status]);
    }

    public function createLead(LeadRequest $request)
    {
        $response['success'] = false;
        $response['message'] = '';

        try {
            $lead = $this->leadService->create($request->all());
            $response['success'] = true;
            $response['message'] = 'Lead added successfully';
        } catch (Exception $e) {
            $response['success'] = false;
            $response['message'] = $e->getMessage();
        }

        return response()->json($response);
    }


    public function updateMetadata(Request $request, $id)
    {
        $lead = $this->leadService->getLead($id);

        if ($request->filled('metadata')) {
            $metadataService = new MetadataService($lead, 'data');
            $metadataService->update($request->metadata);
        }
    }


    /**
     *  Function for convert array to accociative array for front end label and values -- now handle by front end
     */

    // public function mapForLabelAndValues($Array)
    // {
    //     return array_map(function ($value) {
    //         return [
    //             'value' => $value,
    //             'label' => ucwords(str_replace('_', ' ', $value))
    //         ];
    //     }, $Array);
    // }
}
