<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Helper\BriefHelper;
use App\Helper\FileHelper;
use App\Helper\LogsHelper;
use App\Helper\PDFHelper;
use App\Http\Controllers\Controller;
use App\Models\BriefActivity;
use App\Models\BriefLink;
use App\Models\SemBrief;
use App\Models\User;
use App\Models\VisitedBrief;
use App\Services\ApiService;
use App\Services\BinApiService;
use App\Services\BriefActivityService;
use App\Services\IpGeoLocationService;
use App\Services\LogService;
use App\Services\SemBriefService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class SemBriefController extends Controller
{
    private $semBrief, $semBriefService, $apiService, $briefLink, $ipLocationService, $neutrinoApi, $briefActivityService, $logRepository;
    private $logsDirectory = 'semBrief', $linkLongDirectory = 'brief_link';

    public function __construct(SemBrief $semBrief, SemBriefService $semBriefService, ApiService $apiService, BriefLink $briefLink, IpGeoLocationService $ipLocationService,  BinApiService $neutrinoApi, BriefActivityService $briefActivityService, LogService $logRepository)
    {
        $this->semBrief = $semBrief;
        $this->semBriefService = $semBriefService;
        $this->apiService = $apiService;
        $this->briefLink = $briefLink;
        $this->ipLocationService = $ipLocationService;
        $this->neutrinoApi = $neutrinoApi;
        $this->briefActivityService = $briefActivityService;
        $this->logRepository = $logRepository;
    }

    public function index(Request $request)
    {
        $briefs = $this->semBriefService->list($request);

        $page = [
            'title' => 'SEM Briefs List',
            'heading' => 'Briefs',
            'detailRoute' => 'sem.brief-detail.detail',
            'tabs' => [
                [
                    'title' => 'All Briefs',
                    'route' => 'sem.brief.list',
                    'isActive' => true,
                ],
            ],
        ];

        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Brief Form Link", "visible" => true, "fixed" => true],
            ["key" => "name", "component" => "NameColumn", "text" => "Customer Name", "visible" => true],
            ["key" => "phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ["key" => "business_name", "component" => "CompanyColumn", "text" => "Company Name", "visible" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ['key' => "brief_link.salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        // $briefFormLink = 'brief/form';

        // Log the activity before logging out
        // ActivityLogHelper::staticLogActivity(
        //     'visited-log-brief-page',
        //     'Visited logo brief page'
        // );

        return Inertia::render('Admin/Brief/Brief/SEMList', [
            'briefs' => $briefs,
            'page' => $page,
            'columns' => $columns,
        ]);
    }

    public function view(Request $request)
    {
        $isDisabled = true;
        if (Auth::check()) {
            $brief = $this->semBriefService->fetchByEncodedID($request->id, true);
            $trackingApis = FileHelper::readFile($request->id);
        } else {
            $brief = $this->semBriefService->fetchByEncodedID($request->id);
            $trackingApis = [];
        }


        return Inertia::render('SemBriefView', [
            'isDisabled' => $isDisabled,
            'id' => $request->id,
            'brief' => $brief,
            'trackingApis' => $trackingApis,
        ]);
    }

    public function downloadBrief(Request $request, $encoded_id, $fingerprint)
    {
        try {
            $formData = $this->semBriefService->fetchByEncodedID($encoded_id, true);

            $formData['budget'] = $this->stringToArray($formData['budget']);

            $htmlContent = Blade::render('brief.sem_download', [
                'formData' => $formData,
            ]);

            $page = 'A1';
            $pdfOutput = PDFHelper::generatePDF($htmlContent, $page);
            $imageData = $pdfOutput;
            $fileName = $encoded_id . '_image.pdf'; // Adjust the file name as needed

            LogsHelper::log([
                'class_id' => $formData['id'],
                'class_type' => get_class(new SemBrief()),
                'activity' => 'SEM Brief Downloaded by '. $request->user()->email .'.',
                'type' => 'downloaded',
                'created_by' => Auth::user()->id,
                'code' => 200,
                'request' => json_encode(['encoded_id' => $encoded_id, 'fingerprint' => $fingerprint]),
                'response' => response()->json($formData)->content(),
            ], $this->logsDirectory);

            $this->briefActivityService->create($request, $formData['id'], 'sem_brief', 'download', 'Brief Download Successfully', $fingerprint);

            $foundBrief = SemBrief::where('encoded_id', $encoded_id)->first();
            if ($foundBrief) {
                $foundBrief->touch();
            }

            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $fileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }

    public function store(Request $request)
    {
        $rules = [
            'email' => 'required|email|max:255',
            'name' => 'required|string|max:255',
            'designation' => 'required|string|max:255',
            'phone' => 'required|string|max:40',
            'business_name' => 'required|string|max:400',
            'business_idea' => 'required|string|max:400',
            'competitors' => 'required|string|max:400',
            'product_strengths' => 'required|string|max:400',
            'product_weaknesses' => 'required|string|max:400',
            'actions' => 'required|string|max:400',
            'campaign_purpose' => 'required|string|max:255',
            'landing_pages' => 'required|string|max:65535',
            'u_s_p' => 'required|string|max:400',
            'target_places' => 'required|string|max:65535',
            'budget' => 'required|array',

        ];

        $messages = [
            'email.required' => 'Email is required.',
            'email.email' => 'Email must be a valid email address.',
            'name.required' => 'Name is required.',
            'designation.required' => 'Designation is required.',
            'phone.required' => 'Phone number is required.',
            'business_name.required' => 'Business name is required.',
            'business_idea.required' => 'Business idea is required.',
            'competitors.required' => 'List of competitors is required.',
            'product_strengths.required' => 'Product strengths is required.',
            'product_weaknesses.required' => 'Product weaknesses is required.',
            'actions.required' => 'Actions is required.',
            'campaign_purpose.required' => 'Campaign purpose is required.',
            'landing_pages.required' => 'Landing pages is required.',
            'u_s_p.required' => 'Unique Selling Point is required.',
            'target_places.required' => 'Target states is required.',
            'budget.required' => 'Budget is required.',
        ];

        try {
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $nameParts = explode(' ', $request->name, 2);
            $firstName = $nameParts[0];
            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

            if($request->customer_id){
                $existingUser = User::find($request->customer_id);
                if ($existingUser) {
                    $request->merge(['customer_id' => $existingUser->id]);
                    $newEmail = $request->email;

                    if ($existingUser->email !== $newEmail) {
                        $extraEmails = $existingUser->extra_emails ? json_decode($existingUser->extra_emails, true) : [];

                        if (!in_array($newEmail, $extraEmails)) {
                            $extraEmails[] = $newEmail;
                            $existingUser->extra_emails = json_encode($extraEmails);
                            $existingUser->save();
                        }
                    }
                }

            } else {
                $existingUser = User::where('email', $request->email)->first();
                if($existingUser){
                    $request->merge(['customer_id' => $existingUser->id]);
                } else {
                    $user = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->first();

                    $customer = User::create([
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $request->email,
                        'company' => $request->business_name,
                        'phone' => $request->phone,
                        'password' => Hash::make('12345678'),
                        'created_by' => $user->id
                    ]);

                    $customer->assignRole('Customer');

                    $log = array(
                        'activity' => 'Customer was Created',
                        'type' => 'customer.created',
                        'request' => response()->json($customer)->content(),
                        'response' => 201,
                        'code' => response()->json($customer)->status(),
                        'loggable_id' => $customer->id,
                        'created_by' => $customer->id,
                    );
                    $this->logRepository->log('customer', $log);

                    $request->merge(['customer_id' => $customer->id]);
                }

            }


            $semBrief = $this->semBriefService->create($request->all());
            $this->apiService->fetchActivityAndCreateFile($request, $semBrief['encoded_id']);

            $logData = [
                'activity' => 'SEM Brief has been created by ' . $semBrief['email'] . '.',
                'request' => $request,
                'response' => response()->json($semBrief)->content(),
                'created_by' => $request->customer_id,
                'code' => 200,
            ];

            LogsHelper::log(array_merge($logData, [
                'class_id' => $semBrief['brief_link_id'],
                'class_type' => get_class(new BriefLink),
                'type' => 'link.submitted',
            ]), $this->linkLongDirectory);

            LogsHelper::log(array_merge($logData, [
                'class_id' => $semBrief['id'],
                'class_type' => get_class(new SemBrief),
                'type' => 'created',
            ]), $this->logsDirectory);

            $customerLog = array(
                'activity' => 'SEM Brief has been Created',
                'type' => 'customer.brief',
                'request' => response()->json($semBrief)->content(),
                'response' => 201,
                'code' => response()->json($semBrief)->status(),
                'loggable_id' => $request->customer_id,
                'created_by' => $request->customer_id,
            );
            $this->logRepository->log('customer', $customerLog);

            $this->briefActivityService->create($request, $semBrief['id'], 'sem_brief', 'create', 'Brief Created Successfully', $request->fingerprint);

            if($semBrief['encoded_id']){
                $foundBrief = $this->briefLink->where('id', $semBrief['brief_link_id'])->first();
                $foundBrief->customer_id = $request->customer_id;
                $foundBrief->status = "submitted";
                $foundBrief->save();
            }

            $locationResponse = $this->ipLocationService->ipLocation($request->ip());
            $locationData = json_decode($locationResponse, true);
            $locationData['geocode-reverse'] = '';

            $latitude = $locationData['latitude'] ?? null;
            $longitude = $locationData['longitude'] ?? null;

            if ($latitude && $longitude) {
                    $reverseGeocodeData = $this->neutrinoApi->reverseGeocode($latitude, $longitude);
                    $locationData['geocode-reverse'] = $reverseGeocodeData ?? '';
            }

            if($semBrief['id']){
                $this->semBrief->find($semBrief['id'])
                ->update([
                    'location' => $locationData,
                    'device' => $this->neutrinoApi->userAgent($request->header('User-Agent')),
                ]);
            }

            return redirect()->route('sem.brief.success')->with('success', 'Brief created successfully');
        } catch (Exception $e) {

            LogsHelper::log([
                'activity' => 'Error during brief creation.',
                'request' => $request,
                'response' => $e->getMessage(),
                'created_by' => $request->customer_id ?? null,
                'code' => 500,
                'type' => 'link.error',
                'class_id' => $request->brief_link_id,
                'class_type' => get_class(new BriefLink),
            ], $this->linkLongDirectory);

            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function stringToArray($string)
    {
        $array = explode(",", $string);
        return $array;
    }

    public function success()
    {
        $pageProperties = [
            'heading' => 'SEM Brief Questionnaire',
            'description' => ['Your response has been recorded.']
        ];

        return Inertia::render('BriefSuccess', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:sem_briefs,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:sem_briefs,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            if ($request->filled('id')) {
                $brief = SemBrief::findOrFail($request->id);
                $brief->delete();
            } else if ($request->filled('ids')) {
                SemBrief::destroy($request->ids);
            }

            if (is_array($request->ids)) {
                foreach ($request->ids as $id) {
                    $this->briefActivityService->create($request, $id, 'sem_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
                }
            } else {
                $this->briefActivityService->create($request, $request->id, 'sem_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
            }

            return response()->json([
                'success' => true,
                'message' => "Brief(s) successfully deleted.",
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => "An error occurred while deleting brief(s).",
            ], 500);
        }

    }

    public function briefDetail(Request $request){
        $brief = $this->semBrief->with('briefLogs:id,id,brief_id,activity,type,created_by,request,response,pin,code,event,created_at', 'briefLink:id,link,brief_type,salesperson_id,status,visited,created_at', 'customer:id,first_name,last_name,email,phone,address')->findOrFail($request->id);

        $relatedBrief = [];

        $succeededPayments = collect();

        if($brief->customer){
            $relatedBrief = BriefHelper::getRelatedBriefs($brief->customer->id, $request->id, 'sem');

            $customer = User::where('id', '=', $brief->customer->id)->with([
                'payments' => function ($query) {
                    $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                    $query->with([
                        'link' => function ($linkQ) {
                            $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                            $linkQ->with('currencyCountry:id,aplha_code3,code,symbol');
                        }
                    ]);
                    $query->latest('id')->get();
                }
            ])->firstOrFail();

            $customer?->payments->each(function ($payment) use ($succeededPayments) {
                if ($payment->link->status == 2) { // Assuming '2' is the status for successful payments
                    $succeededPayments->push($payment->link);
                }
            });
        }

        $briefDetails = [
            'project_details' => [
                'business_idea' => $brief->business_idea,
                'name' => $brief->name,
                'contact' => $brief->phone,
                'email' => $brief->email,
            ],
            'project_requirements' => [
                'product_strengths' => $brief->product_strengths,
                'competitors' => $brief->competitors,
                'product_weaknesses' => $brief->product_weaknesses,
                'actions' => $brief->actions,
                'campaign_purpose' => $brief->campaign_purpose,
                'landing_pages' => $brief->landing_pages,
                'USP' => $brief->u_s_p,
                'target_places' => $brief->target_places,
                'budget' => $brief->budget,
            ],
        ];

        $briefLogs = LogsHelper::getAllLogs($request->id, $this->logsDirectory);
        $briefLinkLogs = LogsHelper::getAllLogs($brief->briefLink->id, $this->linkLongDirectory);

        $formSubmissionAccuracy = 100;

        $errorLogsCount = 0;
        foreach ($briefLinkLogs['logs'] as $log) {
            if ($log['type'] === 'link.error') {
                $errorLogsCount++;
            }
        }

        $formSubmissionAccuracy -= $errorLogsCount;
        $formSubmissionAccuracy = max($formSubmissionAccuracy, 0);

        $briefActivities = VisitedBrief::where('brief_type', 'sem_brief')->get();

        $activities = [];

        foreach ($briefActivities as $activity) {
            $data = json_decode($activity->data);

            $activities[] = [
                'id' => $activity['id'],
                'ip' => $activity['ip'],
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'userName' => isset($data->userAgent->name) ? $data->userAgent->name : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'createdAt' => $activity['created_at'],
            ];
        }

        $metaData = $brief->briefData()->select('key', 'value')->get();

        $page = [
            'title' => 'SEM Brief Detail',
            'heading' => 'SEM Brief Details',
            'route' => 'sem.brief-detail.detail',
            'briefDirectory' => $this->logsDirectory,
            'metaDataRoute' => 'sem.brief.updateMetadata',
            'deleteRoute' => 'sem.brief.delete',
            'formRoute' => 'frontend.sem.brief.view',
            'activityRoute' => 'sem.brief-detail.activity',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'sem.brief-detail.detail',
                    'isActive' => true  ,
                ],
                // [
                //     'title' => 'Activity',
                //     'route' => 'sem.brief-detail.activity',
                //     'isActive' => false,
                // ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'sem.brief-detail.details',
                //     'isActive' => false,
                // ],
            ],
        ];

        $briefActivitiesLogs = $this->briefActivityService->getBriefActivity($request->id, 'sem_brief');

        $activitiesLogs = [];

        foreach ($briefActivitiesLogs as $activityLog) {
            $data = json_decode($activityLog->data);
            $loginHistory = $activityLog->loginHistory;

            if (isset($loginHistory->user)) {
                $firstName = $loginHistory->user->first_name ?? '';
                $lastName = $loginHistory->user->last_name ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
            } else {
                $fullName = $activityLog['custom_id'];
            }

            $activitiesLogs[] = [
                'id' => $activityLog['id'],
                'custom_id' => $fullName,
                'session_id' => $activityLog['session_id'],
                'brief_id' => $activityLog['brief_id'],
                'brief_type' => $activityLog['brief_type'],
                'ip' => isset($data->ipInfo->ip) ? $data->ipInfo->ip : '',
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'activity' => $activityLog['activity'],
                'activity_desc' => $activityLog['activity_desc'],
                'fingerprint' => $activityLog['fingerprint'],
                'no_of_visits' => $activityLog['no_of_visits'],
                'created_at' => $activityLog['created_at'],
            ];
        }

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        // $this->briefActivityService->createViewActivity($request, $request->id, 'sem_brief', 'view', 'Brief Detail Page Visited');

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'briefDetails' => $briefDetails,
            'brief' => $brief,
            'relatedBrief' => $relatedBrief,
            'metaData' => $metaData,
            'page' => $page,
            'activities' => $activities,
            'logs' => $briefLogs,
            'linkLogs' => $briefLinkLogs,
            'briefID' => $brief->id,
            'activitiesLogs' => $activitiesLogs,
            'authUserRole' => $authUserRole,
            'total_cost' => $succeededPayments->sum('price'),
            'submissionAccuracy' => $formSubmissionAccuracy,
        ]);
    }


    public function briefActivity(Request $request, $id)
    {
        $page = [
            'title' => 'SEM Brief Activity',
            'heading' => 'SEM Brief Activities',
            'route' => 'sem.brief-detail.activity',
        ];

        // $page = [
        //     'title' => 'SEM Brief Activity',
        //     'heading' => 'SEM Brief Activities',
        //     'route' => 'sem.brief-detail.activity',
        //     'tabs' => [
        //         [
        //             'title' => 'Brief Insight',
        //             'route' => 'sem.brief-detail.detail',
        //             'isActive' => false,
        //         ],
        //         [
        //             'title' => 'Activity',
        //             'route' => 'sem.brief-detail.activity',
        //             'isActive' => true,
        //         ],
        //         [
        //             'title' => 'Detail',
        //             'route' => 'sem.brief-detail.details',
        //             'isActive' => false,
        //         ]
        //     ],
        // ];

        $columns = [
            ["key" => "name", "component" => "NameColumn", "text" => "VISITOR ID", "visible" => true, "fixed" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true],
            ["key" => "os", "component" => "OsColumn", "text" => "OS", "visible" => true],
            ["key" => "activity", "component" => "ActivityColumn", "text" => "Activity", "visible" => true],
            ["key" => "fingerprint", "component" => "FingerprintColumn", "text" => "Fingerprint", "visible" => true],
            ["key" => "no_of_visits", "component" => "CountColumn", "text" => "VISITOR COUNT", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        $activities = BriefActivity::select('id', 'custom_id', 'session_id', 'brief_id', 'brief_type', 'data', 'fingerprint', 'activity', 'activity_desc', 'no_of_visits', 'created_at')
            ->with('loginHistory.user')
            ->where('brief_id', $id)
            ->where('brief_type', 'sem_brief')
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $nameParts = array_map('trim', explode(' ', $request->name, 2));
                $firstName = $nameParts[0] ?? null;
                $lastName = $nameParts[1] ?? null;

                $query->whereHas('loginHistory.user', function ($q) use ($firstName, $lastName) {
                    $q->where(function ($q) use ($firstName, $lastName) {
                        if ($firstName && $lastName) {
                            $q->where(function ($q) use ($firstName, $lastName) {
                                $q->where('first_name', 'like', '%' . $firstName . '%')
                                  ->where('last_name', 'like', '%' . $lastName . '%');
                            });
                        }
                        elseif ($firstName) {
                            $q->where('first_name', 'like', '%' . $firstName . '%')
                              ->orWhere('last_name', 'like', '%' . $lastName . '%');
                        }
                    });
                });
            })
            ->when((!empty($request->activitytype)), function ($query) use ($request) {
                $query->where('activity', $request->activitytype);
             })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();


        return Inertia::render('Admin/Brief/Brief/BriefActivityDetail', [
            'page' => $page,
            'activities' => $activities,
            'columns' => $columns,
            'briefID' => $id,
        ]);
    }


    // public function details(Request $request){
    //     $brief = $this->semBriefService->getBrief($request->id);

    //     $page = [
    //         'title' => 'SEM Brief Detail',
    //         'heading' => 'SEM Brief Details',
    //         'route' => 'sem.brief-detail.detail',
    //         'formRoute' => 'frontend.sem.brief.view',
    //         'tabs' => [
    //             [
    //                 'title' => 'Brief Insight',
    //                 'route' => 'sem.brief-detail.detail',
    //                 'isActive' => false,
    //             ],
    //             // [
    //             //     'title' => 'Activity',
    //             //     'route' => 'sem.brief-detail.activity',
    //             //     'isActive' => false,
    //             // ],
    //             // [
    //             //     'title' => 'Detail',
    //             //     'route' => 'sem.brief-detail.details',
    //             //     'isActive' => true,
    //             // ]
    //         ],
    //     ];

    //     $briefDetails = [
    //         'first_column' => [
    //             'client_information' => [
    //                 'email' => $brief->briefLink->customer->email,
    //                 'name' => $brief->briefLink->customer->first_name . ' ' . $brief->briefLink->customer->last_name,
    //                 'designation_(CEO,_Owner,_Manager,etc.)' => $brief->designation,
    //                 'mobile_no' => $brief->briefLink->customer->phone,
    //             ],
    //         ],
    //         'second_column' => [
    //             'project_information' => [
    //                 'registered_business_name' => $brief->business_name,
    //                 'what_makes_you_different_from_your_competitors?' => $brief->u_s_p,
    //                 'list_the_cities,_states,_or_countries_that_you_want_to_target' => $brief->target_places,
    //                 'what_is_your_business_idea_(product/service)' => $brief->business_idea,
    //                 'list_all_your_competitors' => $brief->competitors,
    //                 'what_are_the_strengths_of_your_product/service' => $brief->product_strengths,
    //                 'what_are_the_weaknesses_of_your_product/service' => $brief->product_weaknesses,
    //                 'what_actions_do_you_want_visitors_to_take_on_your_website?' => $brief->actions,
    //                 'what_purpose_do_you_wish_to_achieve_from_this_campaign' => explode(',', $brief->campaign_purpose),
    //                 'what_landing_pages_do_you_want_to_target?' => $brief->landing_pages,
    //                 'please_specify_the_amount_you_wish_to_spend_on_this_campaign_monthly' => explode(',', $brief->budget),
    //             ],
    //         ]
    //     ];

    //     return Inertia::render('Admin/Brief/Brief/BriefDetails', [
    //         'brief' => $brief,
    //         'briefDetails' => $briefDetails,
    //         'page' => $page,
    //         'briefID' => $brief->id,
    //     ]);
    // }

    public function updateMetadata(Request $request){
        try {
            $query = $this->semBrief->findOrFail($request->id);
            BriefHelper::updateMetadata($query, $request->metadata);

            $foundBrief = SemBrief::findOrFail($request->id);
            if ($foundBrief) {
                $foundBrief->touch();
            }

            $this->briefActivityService->create($request, $request->id, 'sem_brief', 'update', 'Metadata Updated Successfully', $request->fingerprint);

        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update metadata'], 500);
        }
    }
}
