<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Helper\FileHelper;
use App\Helper\PDFHelper;
use App\Http\Controllers\Controller;
use App\Models\Brief;
use App\Models\VisitedBrief;
use App\Services\ApiService;
use App\Services\BriefService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Validator;
use App\Helper\ActivityLogHelper;
use App\Helper\BriefHelper;
use App\Models\BriefLink;
use Illuminate\Support\Facades\Http;
use App\Services\IpGeoLocationService;
use App\Services\BinApiService;
use App\Helper\LogsHelper;
use App\Models\BriefActivity;
use App\Models\MobileAppBrief;
use App\Models\SemBrief;
use App\Models\SmmBrief;
use App\Models\User;
use App\Models\VideoBrief;
use App\Models\WebBrief;
use App\Models\LoginHistory;
use App\Services\BriefActivityService;
use App\Services\LogService;
use Illuminate\Support\Facades\Hash;

class BriefController extends Controller
{
    private $brief, $briefService, $apiService, $briefLink, $ipLocationService, $neutrinoApi, $briefActivityService, $logRepository;
    private $logsDirectory = 'brief', $linkLongDirectory = 'brief_link', $mobilelogsDirectory = 'mobileAppBrief', $semlogsDirectory = 'semBrief', $smmlogsDirectory = 'smmBrief', $videologsDirectory = 'videoBrief', $weblogsDirectory = 'webBrief';

    public function __construct(Brief $brief, BriefService $briefService, ApiService $apiService, BriefLink $briefLink, IpGeoLocationService $ipLocationService,  BinApiService $neutrinoApi, BriefActivityService $briefActivityService,  LogService $logRepository)
    {
        $this->brief = $brief;
        $this->briefService = $briefService;
        $this->apiService = $apiService;
        $this->briefLink = $briefLink;
        $this->ipLocationService = $ipLocationService;
        $this->neutrinoApi = $neutrinoApi;
        $this->briefActivityService = $briefActivityService;
        $this->logRepository = $logRepository;
    }

    public function view(Request $request)
    {
        $isDisabled = true;
        if (Auth::check()) {
            $brief = $this->briefService->fetchByEncodedID($request->id, true);
            $trackingApis = FileHelper::readFile($request->id);
        } else {
            $brief = $this->briefService->fetchByEncodedID($request->id);
            $trackingApis = [];
        }

        return Inertia::render('LogoBriefView', [
            'isDisabled' => $isDisabled,
            'id' => $request->id,
            'brief' => $brief,
            'trackingApis' => $trackingApis,
        ]);
    }

   public function index(Request $request)
    {
        $briefs = $this->briefService->list($request);

        $page = [
            'title' => 'Logo Briefs List',
            'heading' => 'Briefs',
            'detailRoute' => 'brief-detail.detail',
            'tabs' => [
                [
                    'title' => 'All Briefs',
                    'route' => 'brief.list',
                    'isActive' => true,
                ],
            ],
        ];

        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Brief Form Link", "visible" => true, "fixed" => true],
            ["key" => "name", "component" => "NameColumn", "text" => "Customer Name", "visible" => true],
            ["key" => "phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ["key" => "business_name", "component" => "CompanyColumn", "text" => "Company Name", "visible" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ["key" => "industry", "component" => "IndustryColumn", "text" => "Industry", "visible" => false],
            ['key' => "brief_link.salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        // $briefFormLink = 'brief/form';

        // Log the activity before logging out
        // ActivityLogHelper::staticLogActivity(
        //     'visited-log-brief-page',
        //     'Visited logo brief page'
        // );

        return Inertia::render('Admin/Brief/Brief/List', [
            'briefs' => $briefs,
            'page' => $page,
            'columns' => $columns,
        ]);
    }

    public function downloadBrief(Request $request, $encoded_id, $fingerprint)
    {
        try {
            $formData = $this->briefService->fetchByEncodedID($encoded_id, true);

            $formData['logo_title'] = $this->stringToArray($formData['logo_title']);
            $formData['font_base_logo'] = $this->stringToArray($formData['font_base_logo']);
            $formData['graphic_base_logo'] = $this->stringToArray($formData['graphic_base_logo']);
            $formData['logo_use'] = $this->stringToArray($formData['logo_use']);
            $formData['business_growth'] = $this->stringToArray($formData['business_growth']);

            $htmlContent = Blade::render('brief.download', [
                'formData' => $formData,
            ]);

            $page = 'A1';
            $pdfOutput = PDFHelper::generatePDF($htmlContent, $page);
            $imageData = $pdfOutput;
            $fileName = $encoded_id . '_image.pdf'; // Adjust the file name as needed

            LogsHelper::log([
                'class_id' => $formData['id'],
                'class_type' => get_class(new Brief()),
                'activity' => 'Logo Brief Downloaded by '. $request->user()->email .'.',
                'type' => 'downloaded',
                'created_by' => Auth::user()->id,
                'code' => 200,
                'request' => json_encode(['encoded_id' => $encoded_id, 'fingerprint' => $fingerprint]),
                'response' => response()->json($formData)->content(),
            ], $this->logsDirectory);

            $this->briefActivityService->create($request, $formData['id'], 'logo_brief', 'download', 'Brief Download Successfully', $fingerprint);

            $foundBrief = Brief::where('encoded_id', $encoded_id)->first();
            if ($foundBrief) {
                $foundBrief->touch();
            }

            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $fileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }

    public function store(Request $request)
    {
        $rules = [
            'email' => 'required|email|max:255',
            'name' => 'required|string|max:255',
            'designation' => 'nullable|string|max:100',
            'phone' => 'nullable|string|max:20',
            'business_name' => 'required|string|max:400',
            'slogan' => 'nullable|string|max:400',
            'industry' => 'nullable|string|max:100',
            'business_description' => 'nullable|string|max:65535',
            'services' => 'nullable|string|max:65535',
            'audience' => 'nullable|string|max:65535',
            'website' => 'nullable|string|max:255',
            'competitor' => 'nullable|string|max:255',
            'logo_title' => 'nullable|array|max:65535',
            'logo_idea' => 'nullable|string|max:65535',
            'logo_message' => 'nullable|string|max:65535',
            'logo_graphical_element' => 'nullable|string|max:65535',
            'logo_customer_likes' => 'nullable|string|max:65535',
            'logo_customer_dont_likes' => 'nullable|string|max:65535',
            'font_base_logo' => 'nullable|array',
            'graphic_base_logo' => 'nullable|array',
            'logo_color' => 'nullable|string|max:400',
            'logo_avoid_color' => 'nullable|string|max:400',
            'logo_font' => 'nullable|string|max:400',
            'logo_look' => 'nullable|string|max:400',
            'logo_use' => 'required|array',
            'business_growth' => 'nullable|array',
            'logo_budget' => 'nullable|string|max:400',
            'contact_time' => 'nullable|string|max:400',
            'anything_else' => 'nullable|string|max:65535',
            'completion_date' => 'required|date_format:Y-m-d',
        ];

        $messages = [
            'email.required' => 'Email is required.',
            'email.email' => 'Email must be a valid email address.',
            'name.required' => 'Name is required.',
            'designation.required' => 'Designation is required.',
            'phone.required' => 'Phone number is required.',
            'business_name.required' => 'Business name is required.',
            'industry.required' => 'Industry is required.',
            'business_description.required' => 'Business description is required.',
            'services.required' => 'Services offered are required.',
            'logo_color.required' => 'Logo color scheme is required.',
            'logo_budget.numeric' => 'Logo budget must be a numeric value.',
            'completion_date.required' => 'Completion date is required.',
            'completion_date.date_format' => 'Completion date must be in the format YYYY-MM-DD.',
        ];

        try {
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {

                LogsHelper::log([
                    'activity' => 'Validation error during brief creation.',
                    'request' => $request,
                    'response' => $validator->errors()->first(),
                    'created_by' => $request->customer_id ?? null,
                    'code' => 403,
                    'type' => 'link.error',
                    'class_id' => $request->brief_link_id,
                    'class_type' => get_class(new BriefLink),
                ], $this->linkLongDirectory);

                return redirect()->back()->withErrors($validator);
            }

            $nameParts = explode(' ', $request->name, 2);
            $firstName = $nameParts[0];
            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

            if($request->customer_id){
                $existingUser = User::find($request->customer_id);
                if ($existingUser) {
                    $request->merge(['customer_id' => $existingUser->id]);
                    $newEmail = $request->email;

                    if ($existingUser->email !== $newEmail) {
                        $extraEmails = $existingUser->extra_emails ? json_decode($existingUser->extra_emails, true) : [];

                        if (!in_array($newEmail, $extraEmails)) {
                            $extraEmails[] = $newEmail;
                            $existingUser->extra_emails = json_encode($extraEmails);
                            $existingUser->save();
                        }
                    }
                }

            } else {
                $existingUser = User::where('email', $request->email)->first();
                if($existingUser){
                    $request->merge(['customer_id' => $existingUser->id]);
                } else {
                    $user = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->first();

                    $customer = User::create([
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $request->email,
                        'company' => $request->business_name,
                        'phone' => $request->phone,
                        'password' => Hash::make('12345678'),
                        'created_by' => $user->id
                    ]);

                    $customer->assignRole('Customer');

                    $log = array(
                        'activity' => 'Customer was Created',
                        'type' => 'customer.created',
                        'request' => response()->json($customer)->content(),
                        'response' => 201,
                        'code' => response()->json($customer)->status(),
                        'loggable_id' => $customer->id,
                        'created_by' => $customer->id,
                    );
                    $this->logRepository->log('customer', $log);

                    $request->merge(['customer_id' => $customer->id]);
                }

            }


            $brief = $this->briefService->create($request->all());
            $this->apiService->fetchActivityAndCreateFile($request, $brief['encoded_id']);

            $logData = [
                'activity' => 'Logo Brief has been submitted by ' . $brief['email'] . '.',
                'request' => $request,
                'response' => response()->json($brief)->content(),
                'created_by' => $request->customer_id,
                'code' => 200,
            ];

            LogsHelper::log(array_merge($logData, [
                'class_id' => $brief['brief_link_id'],
                'class_type' => get_class(new BriefLink),
                'type' => 'link.submitted',
            ]), $this->linkLongDirectory);

            LogsHelper::log(array_merge($logData, [
                'class_id' => $brief['id'],
                'class_type' => get_class(new Brief),
                'type' => 'created',
            ]), $this->logsDirectory);

            $customerLog = array(
                'activity' => 'Logo Brief has been Created',
                'type' => 'customer.brief',
                'request' => response()->json($brief)->content(),
                'response' => 201,
                'code' => response()->json($brief)->status(),
                'loggable_id' => $request->customer_id,
                'created_by' => $request->customer_id,
            );
            $this->logRepository->log('customer', $customerLog);


            $this->briefActivityService->create($request, $brief['id'], 'logo_brief', 'create', 'Brief Created Successfully', $request->fingerprint);

            if($brief['encoded_id']){
                $foundBrief = $this->briefLink->where('id', $brief['brief_link_id'])->first();
                $foundBrief->customer_id = $request->customer_id;
                $foundBrief->status = "submitted";
                $foundBrief->save();
            }

            $locationResponse = $this->ipLocationService->ipLocation($request->ip());
            $locationData = json_decode($locationResponse, true);
            $locationData['geocode-reverse'] = '';

            $latitude = $locationData['latitude'] ?? null;
            $longitude = $locationData['longitude'] ?? null;

            if ($latitude && $longitude) {
                    $reverseGeocodeData = $this->neutrinoApi->reverseGeocode($latitude, $longitude);
                    $locationData['geocode-reverse'] = $reverseGeocodeData ?? '';
            }

            if($brief['id']){
                $this->brief->find($brief['id'])
                ->update([
                    'location' => $locationData,
                    'device' => $this->neutrinoApi->userAgent($request->header('User-Agent')),
                ]);
            }

            return redirect()->route('brief.success')->with('success', 'Brief created successfully');
        } catch (Exception $e) {

            LogsHelper::log([
                'activity' => 'Error during brief creation.',
                'request' => $request,
                'response' => $e->getMessage(),
                'created_by' => $request->customer_id ?? null,
                'code' => 500,
                'type' => 'link.error',
                'class_id' => $request->brief_link_id,
                'class_type' => get_class(new BriefLink),
            ], $this->linkLongDirectory);

            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function stringToArray($string)
    {
        $array = explode(",", $string);
        return $array;
    }

    public function success()
    {
        $pageProperties = [
            'heading' => 'Logo Brief Questionnaire',
            'description' => ['Your response has been recorded.']
        ];

        return Inertia::render('BriefSuccess', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function briefDetail(Request $request)
    {
        $brief = $this->brief->with([
            'briefLogs:id,brief_id,activity,type,created_by,request,response,pin,code,event,created_at',
            'briefLink' => function ($link) {
                $link->select('id','link','brief_type','salesperson_id','status','visited','created_at')->with('salesperson');
            },
            'customer:id,first_name,last_name,email,phone,address'
        ])->findOrFail($request->id);

        $relatedBrief = [];

        $succeededPayments = collect();

        if($brief->customer){
            $relatedBrief = BriefHelper::getRelatedBriefs($brief->customer->id, $request->id, 'logo');

            $customer = User::where('id', '=', $brief->customer->id)->with([
                        'payments' => function ($query) {
                            $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                            $query->with([
                                'link' => function ($linkQ) {
                                    $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                                    $linkQ->with('currencyCountry:id,aplha_code3,code,symbol');
                                }
                            ]);
                            $query->latest('id')->get();
                        }
                    ])->firstOrFail();

                    $customer?->payments->each(function ($payment) use ($succeededPayments) {
                        if ($payment->link->status == 2) { // Assuming '2' is the status for successful payments
                            $succeededPayments->push($payment->link);
                        }
                    });
        }

        $briefDetails = [
            'project_details' => [
                'industry' => ucwords(str_replace('_', ' ', $brief->industry)),
                'business_description' => $brief->business_description,
                'services' => $brief->services,
                'typecase' => explode(',', $brief->logo_title),
                'idea_/_concept' => $brief->logo_idea,
            ],
            'project_requirements' => [
                'identity' => $brief->logo_message,
                'graphical_element' => $brief->logo_graphical_element,
                'preferences' => $brief->logo_customer_likes,
                'avoidances' => $brief->logo_customer_dont_likes,
                'typography' =>  explode(',', $brief->font_base_logo),
                'style' =>  explode(',', $brief->graphic_base_logo),
                'colors_to_use' => $brief->logo_color,
                'colors_to_avoid' => $brief->logo_avoid_color,
                'font_style' => $brief->logo_font,
                'logo_usage' =>  explode(',', $brief->logo_use),
                'strategy' =>  explode(',', $brief->business_growth),
                'note' => $brief->anything_else,
                'completion_date' => $brief->completion_date,
            ],
            'images' => [
                'typography' => [
                    'just_font' => 'backend/images/design-brief/image-1.jpg',
                    'handmade' => 'backend/images/design-brief/image-2.jpg',
                    'initials' => 'backend/images/design-brief/image-3.jpg',
                ],
                'style' => [
                    'abstract_graphic' => 'backend/images/design-brief/image-4.jpg',
                    'silhouette' => 'backend/images/design-brief/image-5.jpg',
                    'illustrated_symbol' => 'backend/images/design-brief/image-6.jpg',
                    'emblem_crests' => 'backend/images/design-brief/image-7.jpg',
                    'mascot' => 'backend/images/design-brief/image-8.jpg',
                ]
            ],
        ];

        $briefLogs = LogsHelper::getAllLogs($request->id, $this->logsDirectory);
        $briefLinkLogs = LogsHelper::getAllLogs($brief->briefLink->id, $this->linkLongDirectory);


        $errorLogsCount = 0;
        foreach ($briefLinkLogs['logs'] as $log) {
            if ($log['type'] === 'link.error') {
                $errorLogsCount++;
            }
        }

        $formSubmissionAccuracy = 100;

        $deduction = 5;
        for ($i = 1; $i <= $errorLogsCount; $i++) {
            $formSubmissionAccuracy -= $deduction;
            $deduction += 3;
        }


        $formSubmissionAccuracy = max(0, $formSubmissionAccuracy);

        $briefActivities = VisitedBrief::where('brief_type', 'logo_brief')->get();

        $activities = [];

        foreach ($briefActivities as $activity) {
            $data = json_decode($activity->data);

            $activities[] = [
                'id' => $activity['id'],
                'ip' => $activity['ip'],
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'userName' => isset($data->userAgent->name) ? $data->userAgent->name : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'createdAt' => $activity['created_at'],
            ];
        }

        $metaData = $brief->briefData()->select('key', 'value')->get();

        $page = [
            'title' => 'Logo Brief Detail',
            'heading' => 'Logo Brief Details',
            'route' => 'brief-detail.detail',
            'briefDirectory' => $this->logsDirectory,
            'metaDataRoute' => 'brief.updateMetadata',
            'deleteRoute' => 'brief.delete',
            'formRoute' => 'frontend.brief.view',
            'activityRoute' => 'brief-detail.activity',
            'projectDetailRoute' => '',
            'projectReqRoute' => '',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'brief-detail.detail',
                    'isActive' => true,
                ],
                // [
                //     'title' => 'Activity',
                //     'route' => 'brief-detail.activity',
                //     'isActive' => false,
                // ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'brief-detail.details',
                //     'isActive' => false,
                // ]
            ],
        ];

        $briefActivitiesLogs = $this->briefActivityService->getBriefActivity($request->id, 'logo_brief');

        $activitiesLogs = [];

        foreach ($briefActivitiesLogs as $activityLog) {
            $data = json_decode($activityLog->data);
            $loginHistory = $activityLog->loginHistory;

            if (isset($loginHistory->user)) {
                $firstName = $loginHistory->user->first_name ?? '';
                $lastName = $loginHistory->user->last_name ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
            } else {
                $fullName = $activityLog['custom_id'];
            }

            $activitiesLogs[] = [
                'id' => $activityLog['id'],
                'custom_id' => $fullName,
                'session_id' => $activityLog['session_id'],
                'brief_id' => $activityLog['brief_id'],
                'brief_type' => $activityLog['brief_type'],
                'ip' => isset($data->ipInfo->ip) ? $data->ipInfo->ip : '',
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'activity' => $activityLog['activity'],
                'activity_desc' => $activityLog['activity_desc'],
                'fingerprint' => $activityLog['fingerprint'],
                'no_of_visits' => $activityLog['no_of_visits'],
                'created_at' => $activityLog['created_at'],
            ];
        }

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        // $this->briefActivityService->createViewActivity($request, $request->id, 'logo_brief', 'view', 'Brief Detail Page Visited');

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'briefDetails' => $briefDetails,
            'brief' => $brief,
            'relatedBrief' => $relatedBrief,
            'metaData' => $metaData,
            'page' => $page,
            'activities' => $activities,
            'logs' => $briefLogs,
            'linkLogs' => $briefLinkLogs,
            'briefID' => $brief->id,
            'activitiesLogs' => $activitiesLogs,
            'authUserRole' => $authUserRole,
            'total_cost' => $succeededPayments->sum('price'),
            'submissionAccuracy' => $formSubmissionAccuracy,
        ]);
    }

    public function briefActivity(Request $request, $id)
    {
        $page = [
            'title' => 'Logo Brief Activity',
            'heading' => 'Logo Brief Activities',
            'route' => 'brief-detail.activity',
        ];

        // $page = [
        //     'title' => 'Logo Brief Activity',
        //     'heading' => 'Logo Brief Activities',
        //     'route' => 'brief-detail.activity',
        //     'tabs' => [
        //         [
        //             'title' => 'Brief Insight',
        //             'route' => 'brief-detail.detail',
        //             'isActive' => false,
        //         ],
        //         [
        //             'title' => 'Activity',
        //             'route' => 'brief-detail.activity',
        //             'isActive' => true,
        //         ],
        //         [
        //             'title' => 'Detail',
        //             'route' => 'brief-detail.details',
        //             'isActive' => false,
        //         ]
        //     ],
        // ];

        $columns = [
            ["key" => "name", "component" => "NameColumn", "text" => "VISITOR ID", "visible" => true, "fixed" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true],
            ["key" => "os", "component" => "OsColumn", "text" => "OS", "visible" => true],
            ["key" => "activity", "component" => "ActivityColumn", "text" => "Activity", "visible" => true],
            ["key" => "fingerprint", "component" => "FingerprintColumn", "text" => "Fingerprint", "visible" => true],
            ["key" => "no_of_visits", "component" => "CountColumn", "text" => "VISITOR COUNT", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        $activities = BriefActivity::select('id', 'custom_id', 'session_id', 'brief_id', 'brief_type', 'data', 'fingerprint', 'activity', 'activity_desc', 'no_of_visits', 'created_at')
            ->with('loginHistory.user')
            ->where('brief_id', $id)
            ->where('brief_type', 'logo_brief')
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $nameParts = array_map('trim', explode(' ', $request->name, 2));
                $firstName = $nameParts[0] ?? null;
                $lastName = $nameParts[1] ?? null;

                $query->whereHas('loginHistory.user', function ($q) use ($firstName, $lastName) {
                    $q->where(function ($q) use ($firstName, $lastName) {
                        if ($firstName && $lastName) {
                            $q->where(function ($q) use ($firstName, $lastName) {
                                $q->where('first_name', 'like', '%' . $firstName . '%')
                                  ->where('last_name', 'like', '%' . $lastName . '%');
                            });
                        }
                        elseif ($firstName) {
                            $q->where('first_name', 'like', '%' . $firstName . '%')
                              ->orWhere('last_name', 'like', '%' . $lastName . '%');
                        }
                    });
                });
            })
            ->when((!empty($request->activitytype)), function ($query) use ($request) {
                $query->where('activity', $request->activitytype);
             })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();


        return Inertia::render('Admin/Brief/Brief/BriefActivityDetail', [
            'page' => $page,
            'activities' => $activities,
            'columns' => $columns,
            'briefID' => $id,
        ]);
    }

    // public function details(Request $request){
    //     $brief = $this->briefService->getBrief($request->id);

    //     $page = [
    //         'title' => 'Logo Brief Detail',
    //         'heading' => 'Logo Brief Details',
    //         'route' => 'brief-detail.detail',
    //         'formRoute' => 'frontend.brief.view',
    //         'tabs' => [
    //             [
    //                 'title' => 'Brief Insight',
    //                 'route' => 'brief-detail.detail',
    //                 'isActive' => false,
    //             ],
    //             // [
    //             //     'title' => 'Activity',
    //             //     'route' => 'brief-detail.activity',
    //             //     'isActive' => false,
    //             // ],
    //             // [
    //             //     'title' => 'Detail',
    //             //     'route' => 'brief-detail.details',
    //             //     'isActive' => true,
    //             // ]
    //         ],
    //     ];

    //     $briefDetails = [
    //         'first_column' => [
    //             'client_information' => [
    //                 'name' => $brief->briefLink->customer->first_name . ' ' . $brief->briefLink->customer->last_name,
    //                 'email' => $brief->briefLink->customer->email,
    //                 'phone' => $brief->briefLink->customer->phone,
    //                 'designation' => $brief->designation,
    //             ],
    //         ],
    //         'second_column' => [
    //             'project_information' => [
    //                 'business_name' => $brief->business_name,
    //                 'tagline_/_slogan_(optional)' => $brief->slogan,
    //                 'industry' => $brief->industry,
    //                 'business_description' => $brief->business_description,
    //                 'services' => $brief->services,
    //                 'target_audience' => $brief->audience,
    //                 'business_website' => $brief->website,
    //                 'competitors' => $brief->competitor,
    //                 'typecase' => explode(',', $brief->logo_title),
    //                 'idea_/_concept' => $brief->logo_idea,
    //                 'identity' => $brief->logo_message,
    //                 'graphical_element' => $brief->logo_graphical_element,
    //                 'preferences' => $brief->logo_customer_likes,
    //                 'avoidances' => $brief->logo_customer_dont_likes,
    //                 'typography' =>  explode(',', $brief->font_base_logo),
    //                 'style' =>  explode(',', $brief->graphic_base_logo),
    //                 'colors_to_use' => $brief->logo_color,
    //                 'colors_to_avoid' => $brief->logo_avoid_color,
    //                 'font_style' => $brief->logo_font,
    //                 'logo_usage' =>  explode(',', $brief->logo_use),
    //                 'strategy' =>  explode(',', $brief->business_growth),
    //                 'logo_budget' => $brief->logo_budget,
    //                 'note' => $brief->anything_else,
    //                 'completion_date:' => $brief->completion_date,
    //             ],
    //         ],
    //         'images' => [
    //             'font_based_logo_options_(select one)' => [
    //                 'just_font' => 'backend/images/design-brief/image-1.jpg',
    //                 'handmade' => 'backend/images/design-brief/image-2.jpg',
    //                 'initials' => 'backend/images/design-brief/image-3.jpg',
    //             ],
    //             'graphic_based_logo_options_(select one)' => [
    //                 'abstract_graphic' => 'backend/images/design-brief/image-4.jpg',
    //                 'silhouette' => 'backend/images/design-brief/image-5.jpg',
    //                 'illustrated_symbol' => 'backend/images/design-brief/image-6.jpg',
    //                 'emblem_crests' => 'backend/images/design-brief/image-7.jpg',
    //                 'mascot' => 'backend/images/design-brief/image-8.jpg',
    //             ]
    //         ],
    //     ];

    //     return Inertia::render('Admin/Brief/Brief/BriefDetails', [
    //         'brief' => $brief,
    //         'briefDetails' => $briefDetails,
    //         'page' => $page,
    //         'briefID' => $brief->id,
    //     ]);
    // }

    /**
     * The function could be as Helper
     */
    public function ipLocation($ip){
        try {
            /**
             * key should be in .env
             */
            $ipData = Http::get("http://api.ipstack.com/$ip?access_key=512d57513320595fdaf8d282ce9aeab2");

            if ($ipData->successful()) {
                $data = $ipData->json();
            } else {
                throw new \ErrorException('Third-party is not responding');
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (Exception $ex) {
            return response()->json([
                'success' => false,
                'code' => $ex->getCode(),
                'message' => $ex->getMessage()
            ]);
        }
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:briefs,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:briefs,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            if ($request->filled('id')) {
                $brief = Brief::findOrFail($request->id);
                $brief->delete();
            } else if ($request->filled('ids')) {
                Brief::destroy($request->ids);
            }

            if (is_array($request->ids)) {
                foreach ($request->ids as $id) {
                    $this->briefActivityService->create($request, $id, 'logo_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
                }
            } else {
                $this->briefActivityService->create($request, $request->id, 'logo_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
            }

            return response()->json([
                'success' => true,
                'message' => "Brief(s) successfully deleted.",
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => "An error occurred while deleting brief(s).",
            ], 500);
        }
    }

    public function updateMetadata(Request $request){
        try {
            $query = $this->brief->findOrFail($request->id);
            BriefHelper::updateMetadata($query, $request->metadata);

            $foundBrief = Brief::findOrFail($request->id);
            if ($foundBrief) {
                $foundBrief->touch();
            }

            $this->briefActivityService->create($request, $request->id, 'logo_brief', 'update', 'Metadata Updated Successfully', $request->fingerprint);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update metadata'], 500);
        }
    }


    public function copyActivity(Request $request){

        LogsHelper::log([
            'class_id' => $request->id,
            'class_type' => $this->getBriefTypeClass($request->brief_type),
            'activity' => $this->formatBriefType($request->brief_type). ' Copied by '. $request->user()->email .'.',
            'type' => 'copied',
            'created_by' => Auth::user()->id,
            'code' => 200,
            'request' => response()->json($request)->content(),
            'response' => json_encode(['brief_id' => $request->id]),
        ], $this->getBriefTypeLogsDir($request->brief_type));

        $this->briefActivityService->create($request, $request->id, $request->brief_type, 'copy', 'Brief Link Copied', $request->fingerprint);
    }

    public function viewActivity(Request $request){

        $briefActivity = $this->briefActivityService->createViewActivity($request, $request->id, $request->brief_type, 'view', 'Brief Form Viewed', $request->fingerprint);

        // $authUser = LoginHistory::with('user')->where('session_id', $briefActivity['session_id'])->first();
        //$userEmailOrCustomId = $authUser && $authUser->user ? $authUser->user->email : $briefActivity['custom_id'];

        if (auth()->check()) {
            $user = auth()->user();
            LogsHelper::log([
                'class_id' => $request->id,
                'class_type' => $this->getBriefTypeClass($request->brief_type),
                'activity' => $this->formatBriefType($request->brief_type). ' Viewed by '. $user->email .'.',
                'type' => 'viewed',
                'created_by' => $user->id,
                'code' => 200,
                'request' => response()->json($request)->content(),
                'response' => json_encode(['brief_id' => $request->id]),
            ], $this->getBriefTypeLogsDir($request->brief_type));
        }

    }

    public function formatBriefType($brief_type)
    {
        $briefTypeFormatted = str_replace('_', ' ', $brief_type);
        $briefTypeFormatted = ucwords($briefTypeFormatted);

        return $briefTypeFormatted;
    }

    public function getBriefTypeLogsDir($brief_type)
    {
        $directoryMap = [
            'logo_brief' => $this->logsDirectory,
            'web_brief' => $this->weblogsDirectory,
            'smm_brief' => $this->smmlogsDirectory,
            'sem_brief' => $this->semlogsDirectory,
            'video_brief' => $this->videologsDirectory,
            'mobile_app_brief' => $this->mobilelogsDirectory,
        ];


        if (!array_key_exists($brief_type, $directoryMap)) {
            throw new \InvalidArgumentException("Invalid Brief type: $brief_type");
        }

        return $directoryMap[$brief_type];
    }

    public function getBriefTypeClass($brief_type)
    {
        $directoryMap = [
            'logo_brief' => get_class(new Brief()),
            'web_brief' => get_class(new WebBrief()),
            'smm_brief' => get_class(new SmmBrief()),
            'sem_brief' => get_class(new SemBrief()),
            'video_brief' => get_class(new VideoBrief()),
            'mobile_app_brief' => get_class(new MobileAppBrief()),
        ];


        if (!array_key_exists($brief_type, $directoryMap)) {
            throw new \InvalidArgumentException("Invalid Brief type: $brief_type");
        }

        return $directoryMap[$brief_type];
    }
}
