<?php

namespace App\Services\Payment\Paypal;

use App\Exceptions\StripeException;
use App\Helper\GeneralHelper;
use App\Helper\Helper\NotificationHelper;
use App\Helper\Helper\StripeHelper;
use App\Repositories\Payment\PaymentInterface;
use App\Services\CustomerService;
use App\Services\LogService;
use App\Services\StripeErrorHandler;
use Carbon\Carbon;
use Exception;

class PaymentService
{
    private $paymentRepository, $logRepository, $customerService;

    function __construct(
        PaymentInterface $payment,
        LogService $log,
        CustomerService $customerService
    ) {
        $this->paymentRepository = $payment;
        $this->logRepository = $log;
        $this->customerService = $customerService;
    }

    public function initiate($customerID, $link, $ip, $gatewayMessage, $status = "")
    {
        $data = [];
        try {
            $data = [
                "customer_id" => $customerID,
                "payment_link_id" => $link->id,
                "price" => $link->price,
                "discount" => $link->discount,
                "currency" => $link->currency,
                "ip" => $ip,
                "converted_amount" => $link->price,
                "last_four" => 0000,
                "comment" => $link->comment,
                "status" => $status != "" ? $status : 3,
                "created_at" => Carbon::now()
            ];

            if (str_contains($link->gateway->gateway, 'three_step')) {
                $data['price'] = ($link->price - 1);
            }

            $payment = $this->paymentRepository->create($data);

            $this->logRepository->log('payment', [
                'activity' => 'Payment is started',
                'type' => 'payment.started',
                'request' => response()->json($data)->content(),
                'response' => response()->json([])->content(),
                'code' => 200,
                'loggable_id' => $payment->id,
                'created_by' => $link->created_by,
            ]);

            return $payment;
        } catch (StripeException $e) {
            throw $e;
        } catch (Exception $e) {
            StripeErrorHandler::handle($e);
        }
    }

    public function addCustomer($customerData, $item_detail)
    {
        try {
            $customer = $this->customerService->create($customerData, $item_detail->id);

            if (!$customer['existing_customer']) {
                $log = array(
                    'activity' => 'Customer was Created',
                    'type' => 'customer.created',
                    'request' => response()->json($customerData)->content(),
                    'response' => response()->json($customer)->content(),
                    'code' => response()->json($customer)->status(),
                    'loggable_id' => $customer['id'],
                    'created_by' => $item_detail->created_by,
                );
                $this->logRepository->log('customer', $log);
            }

            return [
                'internal' => $customer['id']
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function paymentMethod($customer, $link, $orderData, $paymentID, $type = 'paypal')
    {
        $pmID = StripeHelper::generateUniqueID('pm', 10);
        $billing_details['address']['city'] = $customer->city;
        $billing_details['address']['country'] = $customer->country;
        $billing_details['address']['postal_code'] = $customer->zipcode;
        $billing_details['address']['state'] = $customer->state;
        $billing_details['email'] = $customer->email;
        $billing_details['name'] = $customer->first_name . ' ' . $customer->last_name;
        $billing_details['phone'] = $customer->phone;

        $metadata = [
            "token" => $link->token,
            "first_name" => $customer->first_name,
            "last_name" => $customer->last_name,
            "email" => $customer->email,
            "phone" => $customer->phone,
            "company" => $customer->company,
            "address" => $customer->address,
            "city" => $customer->city,
            "state" => $customer->state,
            "country" => $link->currencyCountry->aplha_code3
        ];

        $paymentMethod = [
            "id" => $pmID,
            "object" => "payment_method",
            "billing_details" => $billing_details,
            "created" => Carbon::now()->timestamp,
            "metadata" => $metadata,
            "type" => "card",
            "data" => $orderData
        ];

        if ($type == 'paypal') {
            $paymentMethod['source'] = 'paypal';
            $paymentMethod['paypal'] = $orderData['links'];
        } else {
            $expiry = $orderData['payment_source']['card']['expiry'];

            if (!empty($expiry)) {
                $expiry = explode('-', $expiry);
            }

            $paymentMethod['source'] = 'card';
            $paymentMethod['card']['object'] = $orderData['payment_source']['card'];
            $paymentMethod['card']['brand'] = $orderData['payment_source']['card']['brand'];
            $paymentMethod['card']['country'] = $customer->country;
            $paymentMethod['card']['display_brand'] = $orderData['payment_source']['card']['brand'];
            $paymentMethod['card']['exp_month'] = $expiry[1] ?? '';
            $paymentMethod['card']['exp_year'] = $expiry[0] ?? '';
            $paymentMethod['card']['last4'] = $orderData['payment_source']['card']['last_digits'];

            $avsCode = !empty($orderData['purchase_units'][0]['payments']['captures'][0]['processor_response']['avs_code']) ? $orderData['purchase_units'][0]['payments']['captures'][0]['processor_response']['avs_code'] : null;
            $cvvCode = !empty($orderData['purchase_units'][0]['payments']['captures'][0]['processor_response']['cvv_code']) ? $orderData['purchase_units'][0]['payments']['captures'][0]['processor_response']['cvv_code'] : null;

            $avs_check = $avsCode != null && $avsCode === 'Y' ? 'pass' : 'fail';
            $cvc_check = $cvvCode != null && $cvvCode === 'M' ? 'pass' : 'fail';

            // Checks according to Stripe object
            $paymentMethod['card']['checks']['address_line1_check'] = $avs_check;
            $paymentMethod['card']['checks']['address_postal_code_check'] = $avs_check;
            $paymentMethod['card']['checks']['cvc_check'] = $cvc_check;
        }


        $paymentMethod['status'] = $orderData['status'];

        $authorizedLog = array(
            'activity' => 'Payment is Authorized',
            'type' => 'payment.attach_payment_method',
            'request' => response()->json([
                'payment' => [
                    'id' => $paymentID,
                ],
                'payment_method' => $paymentMethod,
                'created_by' => $link->created_by
            ])->content(),
            'response' => response()->json($paymentMethod)->content(),
            'code' => response()->json($paymentMethod)->status(),
            'loggable_id' => $paymentID,
            'created_by' => $link->created_by,
        );

        $this->logRepository->log('payment', $authorizedLog);

        $customerLog = array(
            'type' => 'customer.payment_method',
            'request' => response()->json($paymentMethod)->content(),
            'response' => response()->json($paymentMethod)->content(),
            'code' => response()->json($paymentMethod)->status(),
            'loggable_id' => $customer->id,
            'created_by' => $link->created_by,
        );

        if ($paymentMethod['source'] == 'paypal') {
            $customerLog['activity'] = 'A new paypal was added';
        } else {
            $customerLog['activity'] = 'A new ' . ucwords($paymentMethod['card']['brand']) . ' card ending in ' . ucwords($paymentMethod['card']['last4']) . ' was added';
        }

        $this->logRepository->log('customer', $customerLog);

        return $paymentMethod;
    }

    public function process($payment, $customer, $link, $orderData, $type = 'paypal')
    {
        try {
            $metadata = [
                "token" => $link->token,
                "first_name" => $customer->first_name,
                "last_name" => $customer->last_name,
                "email" => $customer->email,
                "phone" => $customer->phone,
                "company" => $customer->company,
                "address" => $customer->address,
                "city" => $customer->city,
                "state" => $customer->state,
                "country" => $link->currencyCountry->aplha_code3,
            ];

            $billing_details['address']['city'] = $customer->city;
            $billing_details['address']['country'] = $customer->country;
            $billing_details['address']['postal_code'] = $customer->zipcode;
            $billing_details['address']['state'] = $customer->state;
            $billing_details['email'] = $customer->email;
            $billing_details['name'] = $customer->first_name . ' ' . $customer->first_name;
            $billing_details['phone'] = $customer->phone;

            $paymentIntent = [
                "id" => $orderData['id'],
                "object" => "payment_intent",
                "amount" => $link->price * 100,
                "created" => Carbon::now()->timestamp,
                "currency" => $link->currencyCountry->code,
                "description" => $link->item_name,
                "last_payment_error" => null,
                "metadata" => $metadata,
                "source" => $orderData,
                "statement_descriptor" => "Logo Web Servs LO",
                "status" => $orderData['status'],
            ];

            $intentLog = array(
                'activity' => 'Payment is Initiated',
                'type' => 'payment.initiated',
                'request' => response()->json($link)->content(),
                'response' => response()->json($paymentIntent)->content(),
                'code' => response()->json($paymentIntent)->status(),
                'loggable_id' => $payment->id,
                'created_by' => $link->created_by,
            );

            $this->logRepository->eventStatus(strtolower($orderData['status']), $payment->id, response()->json($paymentIntent)->content(), $link->created_by);

            $intentLog['type'] = 'payment.intent.created';
            $this->logRepository->log('payment', $intentLog, "", 1);

            // Creating Payment Method
            $payment_method = $this->paymentMethod($customer, $link, $orderData, $payment->id, $type);

            return [
                'payment_intent' => $paymentIntent,
                'payment_method' => $payment_method
            ];
        } catch (StripeException $e) {
            throw $e;
        }
    }

    public function generateFile($token, $data, $paymentID, $createdBy)
    {
        try {
            if (!empty($data->cardNo)) {
                $data->cardNo = StripeHelper::maskNumber($data->cardNo);

                StripeHelper::removeKey($data, 'card_number');
                StripeHelper::removeKey($data, 'card_cvc');
                StripeHelper::removeKey($data, 'card_date');
            }

            $intentLog = array(
                'activity' => 'A request to encrypt data is initiated',
                'type' => 'payment.file.initiated',
                'request' => response()->json($data)->content(),
                'response' => response()->json([]),
                'code' => 200,
                'loggable_id' => $paymentID,
                'created_by' => $createdBy,
            );

            $this->logRepository->log('payment', $intentLog, "", 1);

            $file = StripeHelper::createFile($token, $data);

            if ($file) {
                $encrpted = StripeHelper::readFile($token);

                $intentLog = array(
                    'activity' => 'Data has been posted',
                    'type' => 'payment.file.posted',
                    'request' => response()->json($data)->content(),
                    'response' => response()->json($encrpted)->content(),
                    'code' => 200,
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                );

                $this->logRepository->log('payment', $intentLog);

                $intentLog = array(
                    'activity' => 'A new file creation request is completed',
                    'type' => 'payment.file.completed',
                    'request' => response()->json($data)->content(),
                    'response' => response()->json([]),
                    'code' => 200,
                    'loggable_id' => $paymentID,
                    'created_by' => $createdBy,
                );

                $this->logRepository->log('payment', $intentLog, "", 1);
            } else {
                throw new Exception('File not created');
            }
        } catch (Exception $e) {
            $intentLog = array(
                'activity' => 'Failed to store encrypted file',
                'type' => 'payment.file.failed',
                'request' => response()->json($data),
                'response' => response()->json([
                    'code' => $e->getCode(),
                    'message' => $e->getMessage()
                ]),
                'code' => 500,
                'loggable_id' => $paymentID,
                'created_by' => $createdBy,
            );

            $this->logRepository->log('payment', $intentLog);

            return false;
        }
    }
}
