<?php

namespace App\Services;

use App\Repositories\Logs\LogRepository;
use Exception;

class LogsService
{
    private $logRepository;

    public function __construct($storagePath = null)
    {
        $this->logRepository = new LogRepository($storagePath);
    }

    public function log(array $data)
    {
        try {
            return $this->logRepository->log($data);
        } catch (Exception $e) {
            throw new Exception("Failed to log entry: " . $e->getMessage());
        }
    }

    /**
     * Update a log entry by its ID.
     *
     * @param string $logID The ID of the log entry.
     * @param int $classID The class ID associated with the log.
     * @param array $data The data to update in the log entry.
     * @return mixed The updated log entry.
     * @throws Exception If there is an error while updating the log.
     */
    public function updateLog(string $logID, int $classID, array $data)
    {
        try {
            return $this->logRepository->updateLog($logID, $classID, $data);
        } catch (Exception $e) {
            throw new Exception("Failed to update log entry: " . $e->getMessage());
        }
    }

    /**
     * Pin or unpin a log entry by its ID.
     *
     * @param string $logID The ID of the log entry.
     * @param int $classID The class ID associated with the log.
     * @param int $pin The pin status (1 to pin, 0 to unpin).
     * @return mixed The updated log entry with the pin status.
     * @throws Exception If there is an error while pinning/unpinning the log.
     */
    public function pinUnpinLog(string $logID, int $classID, int $pin)
    {
        try {
            return $this->logRepository->pinUnpinLog($logID, $classID, $pin);
        } catch (Exception $e) {
            throw new Exception("Failed to pin/unpin log entry: " . $e->getMessage());
        }
    }

    /**
     * Retrieve a specific log entry by its ID.
     *
     * @param string $logID The ID of the log entry.
     * @param int $classID The class ID associated with the log.
     * @return mixed The log entry.
     * @throws Exception If there is an error while retrieving the log.
     */
    public function getLog(string $logID, int $classID)
    {
        try {
            return $this->logRepository->getLog($logID, $classID);
        } catch (Exception $e) {
            throw new Exception("Failed to retrieve log entry: " . $e->getMessage());
        }
    }

    /**
     * Delete a specific log entry by its ID.
     *
     * @param string $logID The ID of the log entry.
     * @param int $classID The class ID associated with the log.
     * @return bool True if the log entry was successfully deleted, false otherwise.
     * @throws Exception If there is an error while deleting the log.
     */
    public function deleteLog(string $logID, int $classID)
    {
        try {
            return $this->logRepository->deleteLog($logID, $classID);
        } catch (Exception $e) {
            throw new Exception("Failed to delete log entry: " . $e->getMessage());
        }
    }

    /**
     * Retrieve all log entries for a specific class ID.
     *
     * @param int $classID The class ID associated with the logs.
     * @return array An array of log entries.
     * @throws Exception If there is an error while retrieving the logs.
     */
    public function getAllLogs(int $classID)
    {
        try {
            return $this->logRepository->getAllLogs($classID);
        } catch (Exception $e) {
            throw new Exception("Failed to retrieve all logs: " . $e->getMessage());
        }
    }
}
