<?php 
namespace App\Services\Export;

use App\Models\Payment;
use App\Repositories\Export\ExportRepository;
use Exception;

class PaymentsExport {
  protected $exportRepository;

  public function __construct(ExportRepository $exportRepository) {
    $this->exportRepository = $exportRepository;
  }

  public function columns() {
    $exportColumns = array(
        'key' => 'payment',
        'text' => 'Payment Details',
        'columns' => array(
            array('key' => 'id', 'text' => 'id', 'default' => true),
            array('key' => 'price', 'text' => 'amount', 'default' => true),
            array('key' => 'currencyCountry.currency', 'text' => 'currency', 'default' => true),
            array('key' => 'intent_id', 'text' => 'payment_intent_id', 'default' => true),
            array('key' => 'ip', 'text' => 'ip', 'default' => false),
            array('key' => 'charge_id', 'text' => 'charge_id', 'default' => true),
            array('key' => 'converted_amount', 'text' => 'converted_amount', 'default' => true),
            array('key' => 'comment', 'text' => 'comment', 'default' => true),
            array('key' => 'status', 'text' => 'status', 'default' => true),
            array('key' => 'created_at', 'text' => 'datetime', 'default' => true),

        ),
        'related' => array(
            array(
                'key' => 'card',
                'text' => 'Card Details',
                'columns' => array(
                    array('key' => 'paymentMethod.id', 'text' => 'payment_method_id'),
                    array('key' => 'paymentMethod.card.brand', 'text' => 'card'),
                    array('key' => 'paymentMethod.card.brand', 'text' => 'brand'),
                    array('key' => 'paymentMethod.card.exp_month', 'text' => 'card_exp_month'),
                    array('key' => 'paymentMethod.card.exp_year', 'text' => 'card_exp_year', 'default' => false),
                    array('key' => 'paymentMethod.card.last4', 'text' => 'card_last4'),
                    array('key' => 'paymentMethod.card.country', 'text' => 'card_issue_country'),
                ),
            ),
            array(
              'key' => 'customer',
              'text' => 'Customer Details',
              'columns' => array(
                  array('key' => 'customer.stripe_customer_id', 'text' => 'customer_id', 'default' => true),
                  array('key' => 'customer.first_name', 'text' => 'customer_first_name', 'default' => true),
                  array('key' => 'customer.last_name', 'text' => 'customer_last_name', 'default' => true),
                  array('key' => 'customer.email', 'text' => 'customer_email', 'default' => true),
                  array('key' => 'customer.phone', 'text' => 'customer_phone', 'default' => true),
                  array('key' => 'customer.address', 'text' => 'customer_address', 'default' => true),
                  array('key' => 'customer.city', 'text' => 'customer_city', 'default' => false),
                  array('key' => 'customer.state', 'text' => 'customer_state', 'default' => true),
                  array('key' => 'customer.country', 'text' => 'card_country', 'default' => true),
                  array('key' => 'customer.zipcode', 'text' => 'customer_zipcode', 'default' => true),
              ),
          ),

          array(
            'key' => 'invoice',
            'text' => 'Invoice Details',
            'columns' => array(
                array('key' => 'invoice.custom_id', 'text' => 'invoice_id'),
                array('key' => 'invoice.invoice_no', 'text' => 'invoice_no', 'default' => true),
            ),
          ),

          array(
            'key' => 'payment_link',
            'text' => 'Payment Link Details',
            'columns' => array(
                array('key' => 'link.token', 'text' => 'payment_link_token', 'default' => false),
                array('key' => 'link.sale_type', 'text' => 'payment_link_sale_type', 'default' => true),
                array('key' => 'link.gateway.name', 'text' => 'payment_link_gateway', 'default' => true),
                array('key' => 'link.comment', 'text' => 'payment_link_comment'),
                array('key' => 'link.creator.first_name', 'text' => 'payment_link_created_by'),
                array('key' => 'link.created_at', 'text' => 'payment_link_datetime'),
            ),
          ),
        )
        
    );

    $this->exportRepository->headings($exportColumns);
    
    return $exportColumns;
  }

  public function export($data) {
    try {
      $paymentsData = Payment::with([
        'link' => function($link) {
          $link->with([
            'gateway:id,name',
            'creator:id,first_name'
          ]);
        },
        'customer:id,first_name,last_name,email,phone,address,company,city,state,zipcode,country,stripe_customer_id,created_by',
        'invoice:id,custom_id,invoice_no',
        'currencyCountry:id,currency'
      ]);
  
      if(!empty($data['dateRange']) && !empty($data['dateRange']['start']) && !empty($data['dateRange']['end'])) {
        $paymentsData = $paymentsData->whereBetween('created_at', [$data['dateRange']['start'], $data['dateRange']['end']]);
      }
      $payments = $paymentsData->get();
  
      $paymentsExport = $this->exportRepository->process($payments, $data);
  
      $paymentsExport = $this->exportRepository->export();
  
      return $paymentsExport;
    } catch(Exception $e) {
      throw $e;
    }
  }
}