<?php

namespace App\Services;

use App\Repositories\Coupon\CouponRepository;
use Exception;
use Illuminate\Http\Request;

class CouponService
{
    private $couponRepository;

    /**
     * Constructor to inject CouponRepository
     *
     * @param CouponRepository $couponRepository
     */
    public function __construct(CouponRepository $couponRepository)
    {
        $this->couponRepository = $couponRepository;
    }

    /**
     * Retrieve a paginated list of coupons based on request parameters.
     *
     * @param Request $request
     * @return mixed
     * @throws Exception
     */
    public function getCoupons(Request $request)
    {
        try {
            return $this->couponRepository->getCoupons($request);
        } catch (Exception $e) {
            throw new Exception('Failed to fetch coupons: ' . $e->getMessage());
        }
    }

    /**
     * Create a new coupon.
     *
     * @param Request $request
     * @return mixed
     * @throws Exception
     */
    public function create(Request $request)
    {
        try {
            return $this->couponRepository->create($request);
        } catch (Exception $ex) {
            throw new Exception('Failed to create coupon: ' . $ex->getMessage());
        }
    }

    /**
     * Update an existing coupon.
     *
     * @param Request $request
     * @return mixed
     * @throws Exception
     */
    public function update(Request $request)
    {
        try {
            return $this->couponRepository->update($request);
        } catch (Exception $ex) {
            throw new Exception('Failed to update coupon: ' . $ex->getMessage());
        }
    }

    /**
     * Delete a coupon by its ID.
     *
     * @param int $id
     * @return mixed
     * @throws Exception
     */
    public function deleteById($id)
    {
        try {
            return $this->couponRepository->deleteById($id);
        } catch (Exception $e) {
            throw new Exception('Failed to delete coupon: ' . $e->getMessage());
        }
    }

    /**
     * Retrieve a coupon by its ID.
     *
     * @param int $id
     * @return mixed
     * @throws Exception
     */
    public function getCouponById($id)
    {
        try {
            return $this->couponRepository->getCouponById($id);
        } catch (Exception $e) {
            throw new Exception('Failed to fetch coupon: ' . $e->getMessage());
        }
    }

    /**
     * Retrieve detailed information about a coupon.
     *
     * @param int $id
     * @return mixed
     * @throws Exception
     */
    public function getCouponDetail($id)
    {
        try {
            return $this->couponRepository->getCouponDetail($id);
        } catch (Exception $e) {
            throw new Exception('Failed to fetch coupon details: ' . $e->getMessage());
        }
    }

    /**
     * Retrieve a coupon by its name.
     *
     * @param string $name
     * @return mixed
     * @throws Exception
     */
    public function getCouponByName($name)
    {
        try {
            return $this->couponRepository->getCouponByName($name);
        } catch (Exception $e) {
            throw new Exception('Failed to fetch coupon by name: ' . $e->getMessage());
        }
    }
}
