<?php

namespace App\Repositories\SemBrief;

use App\Helper\BriefHelper;
use App\Models\SemBrief;
use App\Services\BriefLogService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use App\Services\LogsService;
use Spatie\Permission\Models\Role;

class SemBriefRepository implements SemBriefInterface
{

    private $semBrief, $briefLogService, $logsService;

    public function __construct(SemBrief $semBrief, BriefLogService $briefLogService)
    {
        $this->semBrief = $semBrief;
        $this->briefLogService = $briefLogService;
        $this->logsService = new LogsService('/logs/briefLogs');
    }

    public function getAllBriefs($request)
    {
        $briefs = $this->semBrief->select('id', 'encoded_id', 'brief_link_id', 'name', 'email', 'phone', 'business_name', 'created_at')
        ->with([
            'briefLink' => function ($link) {
                $link->select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
                    ->with([
                        'salesperson:id,first_name,last_name',
                        'customer:id,first_name,last_name,phone,email,company'
                    ]);
            },
        ])
        ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
            $parsedDate = null;
            if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
            }

            if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
            }
            switch ($request->dateSearchBy) {
                case '>':
                    if (!empty($request->date)) {
                        $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                        $query->whereDate('created_at', '>=', $date);
                    }
                    break;
                case '=':
                    if (!empty($request->date)) {
                        $query->whereDate('created_at', $parsedDate);
                    }
                    break;
                case '<>':
                    if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                        $startDate = Carbon::parse($request->dateBetween['start']);
                        $endDate = Carbon::parse($request->dateBetween['end']);

                        // Adjust time if provided
                        if (!empty($request->timeBetween['start'])) {
                            $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                        }

                        if (!empty($request->timeBetween['end'])) {
                            $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                        }

                        // Adjust timezone
                        if ($request->timezone == 'eastern') {
                            $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                            $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                        }

                        // Adjust timezone
                        if ($request->timezone == 'utc-5') {
                            $startDate->setTimezone('UTC')->addHours(5);
                            $endDate->setTimezone('UTC')->addHours(5);
                        }

                        // Apply date and time range filter
                        $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                    }
                    break;
                case '>=':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                            $query->where('created_at', '>=', $dateTime);
                        } else {
                            $query->whereDate('created_at', '>=', $parsedDate);
                        }
                    }
                    break;
                case '<':
                    if (!empty($request->date)) {
                        if (!empty($request->time)) {
                            $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                            $query->where('created_at', '<', $dateTime);
                        } else {
                            $query->whereDate('created_at', '<', $parsedDate);
                        }
                    }
                    break;
            }
        })
        ->when(!empty($request->name), function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->name . '%');
        })
        ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->where('email', $request->customerEmail);
        })
        ->when(!empty($request->phone), function ($query) use ($request) {
                $query->where('phone', $request->phone);
        })
        ->when(!empty($request->company), function ($query) use ($request) {
                $query->where('business_name', $request->company);
        })
        ->when(!empty($request->status), function ($query) use ($request) {
            $query->whereHas('briefLink', function ($q) use ($request) {
                $q->where('status', $request->status);
            });
        });

        $briefsCount = BriefHelper::getBriefsButtonCount('sem_brief', [
            'title' => 'SEM Brief',
            'count' => $briefs->count(),
            'active' => true,
            'route' => 'sem.brief.list'
        ]);

        $briefs = $briefs->latest('id')->paginate(30)
        ->withQueryString();

        $mergedBriefs = $briefs->toArray();
        $mergedBriefs['buttonsData'] = $briefsCount->all();

        return $mergedBriefs;
    }

    public function getByEncodedID($id, $authenticated = false)
    {
        // Retrieve roles that have the 'BriefForm-Global' permission
        $rolesWithBriefGlobalPermission = Role::whereHas('permissions', function ($query) {
            $query->where('name', 'BriefForm-Global');
        })->pluck('name')->toArray();

        if ($authenticated && Auth::user()->hasAnyRole($rolesWithBriefGlobalPermission)) {
            $briefs = $this->semBrief->where('encoded_id', $id)->firstOrFail();
        } else {
            $briefs = $this->semBrief->excludeColumns(['name', 'email', 'phone'])->where('encoded_id', $id)->firstOrFail();
        }
        return $briefs;
    }

    public function arrayToString($array)
    {
        if (is_array($array)) {
            $string = implode(",", $array);
            return $string;
        }
    }

    public function createBrief($data)
    {
        try {
            $data = (object) $data;

            $semBrief = new $this->semBrief;

            $semBrief->brief_link_id = $data->brief_link_id;
            $semBrief->email = $data->email;
            $semBrief->name = $data->name;
            $semBrief->designation = $data->designation;
            $semBrief->phone = $data->phone;
            $semBrief->business_name = $data->business_name;
            $semBrief->business_idea = $data->business_idea;
            $semBrief->competitors = $data->competitors;
            $semBrief->product_strengths = $data->product_strengths;
            $semBrief->product_weaknesses = $data->product_weaknesses;
            $semBrief->actions = $data->actions;
            $semBrief->campaign_purpose = $data->campaign_purpose;
            $semBrief->landing_pages = $data->landing_pages;
            $semBrief->u_s_p = $data->u_s_p;
            $semBrief->target_places = $data->target_places;
            $semBrief->budget = $this->arrayToString($data->budget);


            $semBrief->customer_id = $data->customer_id ?? Auth::user()->id;

            $semBrief->save();
            $semBrief = $semBrief->refresh();

            // $this->briefLogService->log([
            //     'brief_type' => 'logo_brief',
            //     'brief_id' => $brief->id,
            //     'activity' => 'Logo Brief has been created by ' . $brief->email . '.',
            //     'type' => 'logo_brief.create',
            //     'created_by' => Auth::user()->id,
            // ]);

            // $data = ['class_id' => $brief->id, 'class_type' => get_class($brief), 'activity' => 'Logo Brief has been created by ' . $brief->email . '.', 'type' => 'create', 'request' => '', 'response' => '', 'created_by' => Auth::user()->id, 'event' => null, 'pin' => 0];
            // $this->logsService->log($data);

            return [
                'id' => $semBrief->id,
                'encoded_id' => $semBrief->encoded_id,
                'brief_link_id' => $semBrief->brief_link_id,
                'email' => $semBrief->email ?? '',
                'name' => $semBrief->name ?? '',
                'phone_number' => $semBrief->phone ?? '',
                'business_name' => $semBrief->business_name ?? '',
            ];
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function getBrief($id){
        return $this->semBrief->with([
            'briefLink' => function ($link) {
                $link->with([
                        'salesperson',
                        'customer'
                    ]);
            },
        ])
        ->findOrFail($id);
    }

}
