<?php

namespace App\Repositories\Lead;

use App\Models\Lead;
use App\Models\User;
use App\Services\LogService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;

class LeadRepository implements LeadInterface
{
    private $lead, $logService;

    function __construct(Lead $lead, LogService $logService)
    {
        $this->lead = $lead;
        $this->logService = $logService;
    }

    public function getLeads($request)
    {
        try {
            $leads = $this->lead
                ->select('id', 'name', 'phone', 'email', 'date_time', 'industry', 'description', 'data', 'lead_type', 'lead_source', 'lead_status', 'probability', 'rating', 'contact_medium', 'salesperson', 'created_at')
                ->with('user:id,first_name,last_name')
                ->latest('created_at')
                ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                    $parsedDate = null;
                    if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                        $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                    }

                    if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                        $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                    }
                    switch ($request->dateSearchBy) {
                        case '>':
                            if (!empty($request->date)) {
                                $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                                $query->whereDate('date_time', '>=', $date);
                            }
                            break;
                        case '=':
                            if (!empty($request->date)) {
                                $query->whereDate('date_time', $parsedDate);
                            }
                            break;
                        case '<>':
                            if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                                $startDate = Carbon::parse($request->dateBetween['start']);
                                $endDate = Carbon::parse($request->dateBetween['end']);

                                // Adjust time if provided
                                if (!empty($request->timeBetween['start'])) {
                                    $startDate->setTimeFromTimeString($request->timeBetween['start'] . ":00");
                                }

                                if (!empty($request->timeBetween['end'])) {
                                    $endDate->setTimeFromTimeString($request->timeBetween['end'] . ":59");
                                }

                                // Adjust timezone
                                if ($request->timezone == 'eastern') {
                                    $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                    $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                                }

                                // Adjust timezone
                                if ($request->timezone == 'utc-5') {
                                    $startDate->setTimezone('UTC')->addHours(5);
                                    $endDate->setTimezone('UTC')->addHours(5);
                                }

                                // Apply date and time range filter
                                $query->whereBetween('date_time', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                            }
                            break;
                        case '>=':
                            if (!empty($request->date)) {
                                if (!empty($request->time)) {
                                    $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                    $query->where('date_time', '>=', $dateTime);
                                } else {
                                    $query->whereDate('date_time', '>=', $parsedDate);
                                }
                            }
                            break;
                        case '<':
                            if (!empty($request->date)) {
                                if (!empty($request->time)) {
                                    $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                    $query->where('date_time', '<', $dateTime);
                                } else {
                                    $query->whereDate('date_time', '<', $parsedDate);
                                }
                            }
                            break;
                    }
                })
                ->when(!empty($request->name), function ($query) use ($request) {
                    $query->where('name', 'like', '%' . $request->name . '%');
                })
                ->when(!empty($request->email), function ($query) use ($request) {
                    $query->where('email', $request->email);
                })
                ->when(!empty($request->phone), function ($query) use ($request) {
                    $query->where('phone', $request->phone);
                })
                ->when(!empty($request->status), function ($query) use ($request) {
                    $query->where('lead_status',  strtolower($request->status));
                })
                // ->when(!empty($request->rating), function ($query) use ($request) {
                //     $query->where('rating',  strtolower($request->rating));
                // })
                // ->when(!empty($request->source), function ($query) use ($request) {
                //     $query->where('lead_source',  strtolower($request->source));
                // })
                // ->when(!empty($request->service), function ($query) use ($request) {
                //     $query->where('lead_type',  strtolower($request->lead_type));
                // })
                ->paginate(20)
                ->withQueryString();

            return $leads;
        } catch (Exception $ex) {
            throw $ex;
        }
    }

    public function getLead($id)
    {
        try {
            return $this->lead
                ->select('id', 'name', 'phone', 'email', 'date_time', 'industry', 'description', 'data', 'lead_type', 'lead_source', 'lead_status', 'probability', 'rating', 'contact_medium', 'salesperson', 'created_at')
                ->with([
                    'user:id,first_name,last_name',
                    'quotes',
                    'paymentLogs' => function ($logs) {
                        $logs->latest('created_at');
                    }
                ])->findOrFail($id);
        } catch (Exception $ex) {
            throw $ex;
        }
    }

    public function getLeadCount($status){
        try {
            if (!$status || $status === 'all') {
                return $this->lead->count();
            }
            return $this->lead->where('lead_status', $status)->count();
        } catch (Exception $ex) {
            throw $ex;
        }
    }

    private function calculateProbability($leadStatus)
    {
        switch ($leadStatus) {
            case 'new':
                return 10;
            case 'contacted':
                return 30;
            case 'nurturing':
                return 50;
            case 'qualified':
                return 80;
            case 'unqualified':
                return 0;
            default:
                return 10;
        }
    }

    private function formateLeadData($lead, $request)
    {
        $lead->name = !empty($request['name']) ? $request['name'] : $lead->name;
        $lead->phone = !empty($request['phone']) ? $request['phone'] : $lead->phone;
        $lead->email = !empty($request['email']) ? $request['email'] : $lead->email;
        $lead->date_time = !empty($request['date_time']) ? Carbon::parse($request['date_time'])->format('Y-m-d H:i:s') : $lead->date_time;
        $lead->industry = !empty($request['industry']) ? $request['industry'] : $lead->industry;
        $lead->description = !empty($request['description']) ? $request['description'] : $lead->description;
        $lead->lead_type = !empty($request['lead_type']) ? $request['lead_type'] : $lead->lead_type;
        $lead->lead_source = !empty($request['lead_source']) ? $request['lead_source'] : $lead->lead_source;
        $lead->lead_status = !empty($request['lead_status']) ? $request['lead_status'] : $lead->lead_status;
        $lead->rating = !empty($request['rating']) ? $request['rating'] : $lead->rating;
        $lead->salesperson = !empty($request['salesperson']) ? $request['salesperson'] : $lead->salesperson;
        $lead->salesperson = !empty($request['salesperson']) ? $request['salesperson'] : $lead->salesperson;
        $lead->probability = !empty($request['probability']) ? $request['probability'] : $lead->probability;
        $lead->contact_medium = !empty($request['contact_medium']) ? $request['contact_medium'] : $lead->contact_medium;

        // if (!empty($request['lead_status'])) {
        //     $lead->probability = $this->calculateProbability($request['lead_status']);
        // }

        return $lead;
    }

    public function create($request)
    {
        try {
            $lead = new $this->lead;

            $lead = $this->formateLeadData($lead, $request);

            if ($lead->save()) {
                $this->logService->log('lead', [
                    'activity' => $lead->email . "'s " . ucfirst($lead->lead_type) . " Lead with status " . ucfirst($lead->lead_status) . " and probability " . $lead->probability . "% has been created",
                    'loggable_id' => $lead->id,
                    'type' => 'lead.create',
                    'request' => response()->json($lead)->content(),
                    'response' => response()->json($lead)->content(),
                    'created_by' => Auth::check() ? Auth::user()->id : $lead->salesperson,
                ]);
            }

            return response()->json(['success' => 'Lead created successfully'], 200);
        } catch (Exception $ex) {
            throw $ex;
        }
    }

    public function update($request)
    {
        try {
            $lead = $this->lead->find($request['id']);

            if (!$lead) {
                return response()->json(['error' => 'Lead not found'], 404);
            }

            $lead = $this->formateLeadData($lead, $request);

            if ($lead->save()) {
                $this->logService->log('lead', [
                    'activity' => $lead->email . "'s " . ucfirst($lead->lead_type) . " Lead with status " . ucfirst($lead->lead_status) . " and probability " . $lead->probability . "% has been updated",
                    'loggable_id' => $lead->id,
                    'type' => 'lead.create',
                    'request' => response()->json($lead)->content(),
                    'response' => response()->json($lead)->content(),
                    'created_by' => Auth::user()->id,
                ]);
            }

            return response()->json(['success' => 'Lead updated successfully'], 200);
        } catch (Exception $ex) {
            throw $ex;
        }
    }

    public function delete($id)
    {
        try {
            $lead = $this->lead->findOrFail($id);

            if ($lead->delete()) {
                $this->logService->log('lead', [
                    'activity' => $lead->email . "'s " . ucfirst($lead->lead_type) . " Lead has been deleted",
                    'loggable_id' => $lead->id,
                    'type' => 'lead.delete',
                    'request' => response()->json($lead)->content(),
                    'response' => response()->json($lead)->content(),
                    'created_by' => Auth::user()->id,
                ]);
            }
        } catch (Exception $ex) {
            throw $ex;
        }
    }
}
