<?php

namespace App\Observers;

use App\Models\Team;
use App\Models\UserActivity;
use Illuminate\Support\Facades\Auth;

class TeamObserver
{
    /**
     * Handle the Team "created" event.
     *
     * @param  \App\Models\Team  $team
     * @return void
     */
    public function created(Team $team)
    {
        
        $excludeFields = ['created_by', 'updated_at', 'created_at'];
        $filteredAttributes = array_diff_key(
            $team->getAttributes(),
            array_flip($excludeFields)
        );

        UserActivity::create([
            'model_id' => $team->id,
            'model_type' => get_class($team),
            'activity_desc' => [
                'action' => 'created',
                'details' => $filteredAttributes,
            ],
            'activity_type' => 'create',
            'user_id' => Auth::id(),
        ]);
    }

    /**
     * Handle the Team "updated" event.
     *
     * @param  \App\Models\Team  $team
     * @return void
     */
    public function updated(Team $team)
    {
        $excludeFields = ['created_by', 'updated_by', 'updated_at', 'created_at', 'deleted_at'];

        $original = $team->getOriginal();
        $changes = $team->getChanges();

        $filteredOriginal = array_diff_key($original, array_flip($excludeFields));
        $filteredChanges = array_diff_key($changes, array_flip($excludeFields));

        $updatedFields = array_diff_assoc($filteredChanges, $filteredOriginal);

        if ($updatedFields) {
            UserActivity::create([
                'model_id' => $team->id,
                'model_type' => get_class($team),
                'activity_desc' => [
                    'action' => 'updated',
                    'details' => $updatedFields,
                ],
                'activity_type' => 'update',
                'user_id' => Auth::id(),
            ]);
        }
    }

    /**
     * Handle the Team "deleted" event.
     *
     * @param  \App\Models\Team  $team
     * @return void
     */
    public function deleted(Team $team)
    {
        if ($team->trashed()) {
            UserActivity::create([
                'model_id' => $team->id,
                'model_type' => get_class($team),
                'activity_desc' => [
                    'action' => 'soft-deleted',
                ],
                'activity_type' => 'archive',
                'user_id' => Auth::id(),
            ]);
        }
    }

    /**
     * Handle the Team "restored" event.
     *
     * @param  \App\Models\Team  $team
     * @return void
     */
    public function restored(Team $team)
    {
        UserActivity::create([
            'model_id' => $team->id,
            'model_type' => get_class($team),
            'activity_desc' => [
                'action' => 'restored',
            ],
            'activity_type' => 'unarchive',
            'user_id' => Auth::id(),
        ]);
    }

    /**
     * Handle the Team "force deleted" event.
     *
     * @param  \App\Models\Team  $team
     * @return void
     */
    public function forceDeleted(Team $team)
    {
        UserActivity::create([
            'model_id' => $team->id,
            'model_type' => get_class($team),
            'activity_desc' => [
                'action' => 'deleted',
            ],
            'activity_type' => 'delete',
            'user_id' => Auth::id(),
        ]);
    }
}
