<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'password',
        'phone',
        'city',
        'company',
        'address',
        'state',
        'zipcode',
        'country',
        'image',
        'stripe_customer_id',
        'stripe',
        'stripe_pm_id',
        'stripe_payment_method',
        'dashboard_data',
        'created_by',
        'target',
        'isBlocked',
        'facebook',
        'instagram',
        'linkedin',
        'password_expire',
        'two_step',
        'two_step_created',
        'two_step_checked',
        'google2fa_secret',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    public function paymentLogs() {
        return $this->morphMany(PaymentLog::class, 'loggable');
    }

    public function quotes() {
        return $this->morphMany(Quotes::class, 'commentable');
    }

    public function countryData() {
        return $this->hasOne(CountryCurrencies::class, 'id', 'country');
    }

    public function payments() {
        return $this->hasMany(Payment::class, 'customer_id', 'id');
    }

    public function invoices()
    {
        return $this->hasMany(Invoice::class, 'customer_id', 'id');
    }

    public function teams()
    {
        return $this->belongsToMany(Team::class, 'user_team', 'user_id', 'team_id');
    }

    public function boundCustomers()
    {
        return $this->belongsToMany(User::class, 'customer_bindings', 'customer_a_id', 'customer_b_id')
                    ->withTimestamps()
                    ->withPivot('created_at', 'updated_at');
    }

    public function allBoundCustomers()
    {
        // Fetch all customer IDs bound to this customer, using recursive SQL logic
        $boundCustomerIds = DB::select("
            WITH RECURSIVE bound_customers AS (
                SELECT customer_a_id AS customer_id FROM customer_bindings WHERE customer_b_id = ?
                UNION
                SELECT customer_b_id AS customer_id FROM customer_bindings WHERE customer_a_id = ?
                UNION
                SELECT cb.customer_b_id AS customer_id
                FROM customer_bindings cb
                JOIN bound_customers bc ON cb.customer_a_id = bc.customer_id
                WHERE cb.customer_b_id != ?
            )
            SELECT DISTINCT customer_id FROM bound_customers
        ", [$this->id, $this->id, $this->id]);

        // Convert results to a simple array of customer IDs
        $boundCustomerIds = array_map(function ($record) {
            return $record->customer_id;
        }, $boundCustomerIds);

        // Include the current customer's own ID
        $boundCustomerIds[] = $this->id;

        return array_unique($boundCustomerIds);
    }

    public function allPayments()
    {
        $boundCustomerIds = $this->allBoundCustomers();
        $boundCustomerIds[] = $this->id; // Include this customer's own ID

        return Payment::whereIn('customer_id', $boundCustomerIds)->get();
    }

    static function topSpendCustomers($startDate = "", $endDate = "", $userIDs = "", $userInclusion = false,) {
        $topCustomers = self::select('id', 'first_name', 'last_name', 'email', 'stripe_customer_id');
        $topCustomers = $topCustomers->withSum('payments', 'price');
        $topCustomers = $topCustomers->whereHas('payments', function($query) use ($startDate, $endDate, $userIDs, $userInclusion) {
            $query->whereHas('link', function ($innerQuery) use ($userIDs, $userInclusion) {
                if($userIDs != "") {
                    if(gettype($userIDs) == 'array' || gettype($userIDs) == 'object') {
                        $inclusion = $userInclusion ? 'whereIn' : 'whereNotIn';
                        $innerQuery->{$inclusion}('created_by', $userIDs);
                    } else {
                        if($userInclusion) {
                            $innerQuery->where('created_by', $userIDs);
                        } else {
                            $innerQuery->where('created_by', '<>', $userIDs);
                        }
                    }
                }
            });

            if($startDate != "") {
                $query->whereDate('created_at', '>=', $startDate);
            }
            
            if($endDate != "") {
                $query->whereDate('created_at', '<=', $endDate);
            }
        });

        $topCustomers = $topCustomers->orderByDesc('payments_sum_price')->limit(3)->get();
        return $topCustomers;
    }
}
