<?php

namespace App\Jobs;

use Carbon\Carbon;
use App\Models\Payment;
use App\Models\PaymentGateway;
use App\Services\Payment\Stripe\Refund;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class RefundListJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        try {
            $twoDaysAgo = Carbon::now()->subDays(3)->toDateString();
            $paymentsQuery = Payment::select('id', 'payment_link_id', 'price', 'intent_id')
                ->with(['link' => function ($link) {
                    $link->select('id', 'payment_gateway', 'currency', 'created_by');
                    $link->with(['gateway:id,name,secret_key']);
                }])
                ->whereNotNull('intent_id')
                ->whereDate('created_at', $twoDaysAgo)
                ->where('status', 1);
        
            Log::info('Job Run: ' . $paymentsQuery->toSql(), $paymentsQuery->getBindings());
        
            if ($paymentsQuery->exists()) {
                $payments = $paymentsQuery->get();
                Log::info('Payments exists - ' . count($payments));
        
                foreach ($payments as $payment) {
                    Log::info('Dispatching job for payment ID: ' . $payment->id);
                    CheckPaymentRefunded::dispatch(
                        $payment->id,
                        $payment->intent_id,
                        $payment->price,
                        $payment->link->gateway->secret_key,
                        $payment->link->created_by
                    );
                }
            } else {
                Log::info('No payments found for the specified criteria.');
            }
        } catch (Exception $e) {
            Log::error($e->getMessage());
        } 
    }
}
