<?php

namespace App\Http\Requests;

use Carbon\Carbon;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Arr;

class StripePaymentRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $currentYear = Carbon::now()->format('y');
        $cardNo = str_replace(' ', '', $this->input("cardNo"));
        
        return [
            'firstname' => 'required',
            'lastname' => 'required',
            'clientemail' => ['required', 'email'],
            'phonenum' => 'required',
            'address' => 'required',
            'companyname' => 'required',
            'country' => 'required',
            'statename' => 'required',
            'city' => 'required',
            'zipcode' => 'required',
            'cardNo' => ['required', 'digits_between:14,19'],
            'month' => ['required', 'digits:2'],
            'cardValidity' => ['required', function ($attribute, $value, $fail) {
              if (!$value) {
                $fail("The card number is invalid. Check the card details or use a different card.");
              }
            }],
            'year' => [
              'required', 'digits:2',
              function ($attribute, $value, $fail) use ($currentYear) {
                if ($currentYear > $value) {
                  $fail("The card's expiration year is incorrect.");
                }
              },
            ],
            'cvc' => [
              'required', 'digits_between:3,4',
              function ($attribute, $value, $fail) use ($cardNo) {
                if (strlen((string)$cardNo) === 15 && strlen((string)$value) !== 4) {
                  $fail("Check the card's security code or use a different card.");
                }
              },
            ],
        ];
    }

    public function messages() {
        return [
            'cardNo.required' => 'The card number is invalid. ',
            'cardNo.digits_between' => 'The card number is invalid. ',
            'cardValidity.required' => "The card number is invalid. Check the card details or use a different card.",
            'month.required' => "The card's expiration month is incorrect.",
            'month.digits' => "The card's expiration month is incorrect.",
            'year.digits' => "The card's expiration year is incorrect.",
            'year.required' => "The card's expiration year is incorrect.",
            'cvc.required' => "Check the card's security code or use a different card.",
            'cvc.digits_between' => "Check the card's security code or use a different card.",
        ];
    } 

    function validateCardNumber($cardNumber)
    {
        // Remove any non-digit characters from the card number
        $cleanedCardNumber = preg_replace('/\D/', '', $cardNumber);

        // Convert the card number to an array of digits
        $digits = str_split($cleanedCardNumber);

        // Double every second digit starting from the right
        for ($i = count($digits) - 2; $i >= 0; $i -= 2) {
            $digits[$i] *= 2;
            if ($digits[$i] > 9) {
                $digits[$i] -= 9;
            }
        }

        // Sum all the digits
        $sum = array_sum($digits);

        // Check if the sum is divisible by 10
        return $sum % 10 === 0;
    }
}
