<?php

namespace App\Http\Middleware;

use App\Helper\GeneralHelper;
use App\Helper\Helper\StripeHelper;
use App\Helper\PaymentHelper;
use App\Models\CountryCurrencies;
use App\Models\Payment;
use App\Models\PaymentLink;
use App\Services\BinApiService;
use App\Services\CustomerService;
use App\Services\IpGeoLocationService;
use App\Services\Payment\PaymentService;
use Carbon\Carbon;
use Closure;
use Illuminate\Http\Request;

class CustomRadar
{
    protected $neutrinoApi, $geoLocation;
    protected $customerService, $paymentService;

    function __construct(BinApiService $binApiService, IpGeoLocationService $geoLocation, CustomerService $customerService, PaymentService $paymentService)
    {
        $this->neutrinoApi = $binApiService;
        $this->geoLocation = $geoLocation;

        $this->customerService = $customerService;
        $this->paymentService = $paymentService;
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Check IP address
        $ip = $request->ip();
        $userAgent = $request->header('User-Agent');

        $checkHighValueTransaction = $this->highValuedTransaction($request->itemprice) ? 10 : 0;
        
        // High Alerts 
        /* 
            - Geolocation and IP address
            - Payment method detail
            - Card verification 
            - Behavior Analysis   
        */
        $checkIP = $this->checkIP($ip, $vpnLookup = false) ? 10 : 0;
        // verify card detail
        $card8Digits = substr($request->cardNo, 0, 8);
        $cardVerification = $this->paymentMethodCheck($card8Digits, $ip) ? 10 : 0;
        
        // Low Alerts
        /*
            - Device Finger
            - Velocity check
            - Email Address
            - Frequency of transaction
            - Shipping Address & Billing Address
        */

        $checkDevice = $this->checkDeviceScore($userAgent) ? 10 : 0;
        $emailAddress = $this->checkEmailAddressScore($request->clientemail) ? 10 : 0;

        $countryCode = CountryCurrencies::select('id','aplha_code2','aplha_code3')->where('country', $request->country)->first();
        $shippingAndBillingAddress = 0;
        /* $shippingAndBillingAddress = $this->geolocateVerify(
            address: $request->address,
            city: $request->city,
            state: $request->statename, 
            postalCode: $request->zipcode, 
            countryCode: $countryCode->aplha_code3
        ) ? 10 : 0; */

        $last_four = substr($request->cardNo, -4);
        $velocityAndFrequencyCheck = $this->velocityTransactionsScore($request->clientemail, $ip, $last_four, $timeInSeconds = 120, $pointWeight = 40);

        $scores = [
            'high_value_transaction' => $checkHighValueTransaction,
            'ip_blacklist' => $checkIP,
            'card_verification' => $cardVerification,
            'device_verification' => $checkDevice,
            'email_verification' => $emailAddress,
            'velocity_and_frequency' => $velocityAndFrequencyCheck,
        ];

        // If conditions are not met, redirect to failed payment page
        if ($this->scoreAnalyse($scores) > 60) {

            $formData = (array) $request->all();
            $paymentDetails = (object) PaymentHelper::itemArray($formData);
            $item_detail = PaymentLink::where("token", "=", $paymentDetails->token)
                ->with(
                    'currencyCountry:id,aplha_code2,aplha_code3,code,symbol',
                    'gateway:id,name,gateway,public_key,secret_key,statement_descriptor,environment'
                )->first();

            $paymentDetails->brand_descriptor = $item_detail->gateway->statement_descriptor;
            $paymentDetails->currency_country = $item_detail->currencyCountry;
            $paymentDetails->link_id = $item_detail->id;

            $card8Digits = substr($paymentDetails->cardNo, 0, 8);
            $paymentDetails->cardNo = StripeHelper::maskNumber($paymentDetails->cardNo);

            StripeHelper::removeKey($paymentDetails, 'card_number');
            StripeHelper::removeKey($paymentDetails, 'card_cvc');
            StripeHelper::removeKey($paymentDetails, 'card_date');

            $customer = $this->customerService->create($paymentDetails, $item_detail->id);

            $payment = $this->paymentService->initiate($customer['id'], $item_detail, $request->ip(), $item_detail->comment, "");
            
            $device = $this->neutrinoApi->userAgent($request->header('User-Agent'));
            $location = $this->geoLocation->ipLocation($request->ip());
            $binLookup = $this->neutrinoApi->binLookup($card8Digits, $request->ip());

            $updatePayment = [
                'location' => $location->body(),
                'device' => response()->json($device)->content(),
                'bin' => response()->json($binLookup)->content(),
                'status'=> 6
            ];

            $payment->update($updatePayment);
            $payment->save();

            $request->merge(['payment_id' => $payment->id]);
            $request->merge(['error' => [
                "code" => "blocked_customer",
                "message" => "Customer blocked by custom radar.",
                "type" => "customer_error",
                "score" => $this->scoreAnalyse($scores),
                "statusCode" => 401
            ]]);
        }


        return $next($request);
    }

    function scoreAnalyse($scores) {
        $totalScore = 0;

        foreach ($scores as $score) {
            $totalScore += $score;
        }

        return ($totalScore / 100) * 100;
    }

    function highValuedTransaction($amount)
    {
        if ($amount > 10000) {
            return true;
        }

        return false;
    }

    function checkIP($ip, $vpnLookup = false)
    {
        $apiResponse = $this->neutrinoApi->ipBlocklist($ip, $vpnLookup);
        $status = false;

        if ($apiResponse) {
            $types = ['is-proxy', 'is-tor', 'is-malware', 'is-spyware', 'is-dshield', 'is-hijacked', 'is-spider', 'is-bot', 'is-spam-bot', 'is-exploit-bot'];

            foreach($types as $type) {
                if(!empty($apiResponse[$type])) {
                    $status = true;
                }
            }
        }
        
        return $status;
    }

    function checkDeviceScore($userAgent)
    {
        $apiResponse = $this->neutrinoApi->userAgent($userAgent);
        return false;

        // need to fix this
        dd($apiResponse);
        if($apiResponse) {
            if($apiResponse['valid']) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    function checkEmailAddressScore($email)
    {
        $apiResponse = $this->neutrinoApi->emailVerify($email);

        if($apiResponse && !empty($apiResponse['valid'])) {
            return $apiResponse['valid'];
        } else {
            return false;
        }
    }

    // Get recent payments which occurs rapidly in desired seconds (default: 120 seconds)
    function checkRecentPayments($customerEmail, $ip, $last_four, $timeInSeconds = 120)
    {
        $currentDateTime = Carbon::now();
        $pastDateTime = $currentDateTime->subSeconds($timeInSeconds);

        $payments = Payment::select('id','customer_id','payment_link_id','price','currency','ip','charge','refund','intent','paymentMethod','last_four','device','location','bin','status','created_at')
                        ->with('link:id,token,valid_till,item_name,price,currency,status')
                        ->where('ip', $ip)
                        ->orWhere('last_four', $last_four)
                        ->orWhereHas('customer', function($customer) use ($customerEmail) {
                            $customer->select('id','first_name','last_name','email','address','city','state','country');
                            $customer->where('email', $customerEmail);
                        })
                        ->whereBetween('created_at', [$pastDateTime, $currentDateTime])
                        ->limit(4)
                        ->get();

        return $payments;
    }

    function velocityTransactionsScore($customerEmail, $ip, $last_four, $timeInSeconds = 120, $pointWeight = 40) {
        $payments = $this->checkRecentPayments($customerEmail, $ip, $last_four, $timeInSeconds);
        $totalPoints = 0;

        if(count($payments) > 0) {
            foreach($payments as $payment) {
                $checkIP = $this->checkIP($payment->ip) ? 10 : 0;
                $checkHighValueTransaction = $this->highValuedTransaction($payment->price) ? 10 : 0;
                
                $userAgent = GeneralHelper::decodeJSON($payment->device);
                $ua = !empty($userAgent?->ua) ? $userAgent?->ua : (!empty($userAgent?->userAgent) ? $userAgent?->userAgent : false);
                $checkDevice = ($ua) ? ($this->checkDeviceScore($ua) ? 10 : 0) : 0;

                if($payment?->customer?->email) {
                    $emailAddress = $this->checkEmailAddressScore($payment->customer->email) ? 10 : 0;
                    
                    $countryCode = CountryCurrencies::select('id','aplha_code2','aplha_code3')->where('country', $payment->customer->country)->first();

                    $shippingAndBillingAddress = 0;
                    /* $shippingAndBillingAddress = $this->geolocateVerify(
                        address: $payment->customer->address,
                        city: $payment->customer->city,
                        state: $payment->customer->statename, 
                        postalCode: $payment->customer->zipcode, 
                        countryCode: $countryCode->aplha_code3
                    ) ? 10 : 0; */
                } else {
                    $emailAddress = 0;
                    $shippingAndBillingAddress = 0;
                }

                $totalPoints += (($checkIP + $checkHighValueTransaction + $checkDevice + $emailAddress + $shippingAndBillingAddress)/5)*$pointWeight;
            }
            return ($totalPoints / (count($payments) * $pointWeight) ) / $pointWeight;
        } else {
            return 0;
        }
    }

    function cardVerificationScore()
    {
    }

    function paymentMethodCheck($first8Digits, $ip)
    {
        $apiResponse = $this->neutrinoApi->binLookup($first8Digits, $ip);

        if($apiResponse && !empty($apiResponse['valid'])) {
            return $apiResponse['valid'];
        } else {
            return false;
        }
        
    }

    function geolocateVerify($address, $houseNumber = "", $street = "", $city, $county="", $state, $postalCode, $countryCode, $languageCode ='en', $fuzzySearch= false) {
        $apiResponse = $this->neutrinoApi->geolocate($address, $houseNumber, $street, $city, $county, $state, $postalCode, $countryCode, $languageCode, $fuzzySearch);
        
        if($apiResponse && !empty($apiResponse['found'])) {
            return $apiResponse['found'] > 0 ? false : true;
        } else {
            return false;
        }
    }
}
