<?php

namespace App\Http\Controllers\Admin;

use App\Helper\Helper\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\PaymentLog;
use App\Models\Team;
use App\Services\TeamService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;

class TeamController extends Controller
{
    private $teamService;

    function __construct(TeamService $teamService)
    {
        $this->teamService = $teamService;
    }

    public function list(Request $request)
    {
        $page["title"] = "Teams List - TeamModule";

        $columns = array(
            array("key" => "name", "component" => 'NameColumn', "text" => "Name", "visible" => true, "fixed" => true),
            array("key" => "created_at", "component" => 'CreatedAtColumn', "text" => "Created", "visible" => true),
            array("key" => "updated_at", "component" => 'UpdatedAtColumn', "text" => "Updated", "visible" => true),
        );

        $statuses = ['all', 'available', 'archived'];

        $teams = Team::query()
            ->select('id', 'name', 'description', 'target', 'color', 'created_by', 'updated_by', 'status', 'created_at', 'updated_at', 'deleted_at')
            ->when(!empty($request->status_all), function ($query) use ($request, $statuses) {
                $currentStatus = 0;
                foreach ($statuses as $key => $status) {
                    if ($status == $request->status_all) {
                        $currentStatus = $key;
                    }
                }
                if ($currentStatus == 2) {
                    $query->onlyTrashed();
                } elseif ($currentStatus == 1) {
                    $query->where('status', 1);
                } else {
                    $query->withTrashed();
                }
            })
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        $teamSums = Team::statusCount($statuses);

        return Inertia::render('Admin/Team/List', [
            'columns' => $columns,
            'page' => $page,
            'teams' => collect($teamSums)->merge($teams),
        ]);
    }

    public function create()
    {
        $page["title"] = "Create Team Catalog - TeamModule";

        return Inertia::render('Admin/Team/Create', [
            'page' => $page,
        ]);
    }

    public function generate(Request $request)
    {
        try {
            $rules =  [
                'name' => 'required',
                'color' => 'required',
                'target' => 'required|numeric',
            ];

            $CustomMessages = [
                'name.required' => 'Team Name is required.',
                'color.required' => 'Team Color Code is required.',
                'target.required' => 'Team Target is required.',
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.team.list')
                    ->withErrors($validator)
                    ->withInput();
            }

            $request['created_by'] = Auth::id();
            $teamService = $this->teamService->create($request->all());


            if ($teamService["success"] == true) {
                $invoice_id = $teamService['id'];
                $invoice_details =  Team::find($invoice_id);
                $log = new PaymentLog();
                $log->activity = 'A New Team ' . $invoice_details->name . "'s Created";
                $log->type = 'created';
                $log->request = response()->json($request->all())->content();
                $log->response = response()->json($request->all())->content();
                $log->code = 200;
                $log->created_by = $invoice_details->created_by;
                $log->loggable()->associate($invoice_details);
                $log->save();

                NotificationHelper::notify('team', 'A New Team ' . $invoice_details->name . "'s Created", [
                    'id' => $invoice_id,
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($request->all())->content(),
                    'message' => 'Team Added Successful',
                  ], Auth::user()->id);
                Session::flash('success', true);
                Session::flash('message', 'New Team Created.');
                return Redirect::route('admin.team.list');
            }
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());
            return Redirect::route('admin.team.list');
        }
    }

    public function getTeam($teamID)
    {
        $page["title"] = "Edit Team Catalog - TeamModule";

        $data['success'] = true;
        try {
            $team = $this->teamService->getTeamById($teamID);

            // return Inertia::render('Admin/Team/Edit', [
            //     'page' => $page,
            //     'team' => $team,
            // ]);

            return response()->json([
                'page' => $page,
                "team" => $team
            ]);
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function update(Request $request, $teamID)
    {
        try {
            $rules = [
                'name' => 'required',
                'color' => 'required',
                'target' => 'required|numeric',
            ];

            $customMessages = [
                'name.required' => 'Team Name is required.',
                'color.required' => 'Team Color Code is required.',
                'target.required' => 'Target is required.',
                'target.numeric' => 'Target must be a number.',
            ];

            $validator = Validator::make($request->all(), $rules, $customMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.team.detail', $teamID)
                    ->withErrors($validator)
                    ->withInput();
            }
            $team = Team::findOrFail($teamID);
            $currentName = $team->name;
            $teamUpdated = $team->update([
                'name' => $request->input('name'),
                'description' => $request->input('description'),
                'target' => $request->input('target'),
                'color' => $request->input('color'),
                'updated_by' => Auth::id(),
            ]);

            if ($teamUpdated) {
                $log = new PaymentLog();
                $log->activity = 'Team "' . $currentName . '" updated to "' . $team->name  . '"';
                $log->type = 'updated';
                $log->request = response()->json($request->all())->content();
                $log->response = response()->json($request->all())->content();
                $log->code = 200;
                $log->created_by = Auth::id();
                $log->loggable()->associate($team);
                $log->save();
                NotificationHelper::notify('team', 'Team "' . $currentName . '" updated to "' . $team->name  . '"', [
                    'id' => $teamID,
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($request->all())->content(),
                    'message' => 'Team Updated Successful',
                  ], Auth::user()->id);

                Session::flash('success', true);
                Session::flash('message', 'Team Catalog Updated.');
            } else {
                // Flash error message to the session if the update fails
                Session::flash('success', false);
                Session::flash('message', 'Failed to update team catalog.');
            }

            // Flash success message to the session
            Session::flash('success', true);
            Session::flash('message', 'Team Catalog Updated.');

            // Redirect to the team detail page
            return Redirect::route('admin.team.detail', $teamID);
        } catch (Exception $ex) {
            // Handle exceptions
            Session::flash('success', false);
            Session::flash('message', $ex->getMessage());

            // Redirect to the team detail page
            return Redirect::route('admin.team.detail', $teamID);
        }
    }

    public function archived($teamID)
    {
        $data['success'] = true;
        try {
            $teamService = $this->teamService->archiveTeam($teamID);
            Session::flash('success', true);
            Session::flash('message', $teamService["message"]);
            return Redirect::route('admin.team.list');
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function delete($teamID)
    {
        $data['success'] = true;
        try {
            $teamService = $this->teamService->deleteTeam($teamID);

            if ($teamService["success"] == true) {
                Session::flash('success', true);
                Session::flash('message', $teamService["message"]);
                return Redirect::route('admin.team.list');
            }
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function restore($teamID)
    {
        $data['success'] = true;
        try {
            $teamService = $this->teamService->restoreTeam($teamID);
            Session::flash('success', true);
            Session::flash('message', $teamService["message"]);
            return Redirect::route('admin.team.list');
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function detail($teamID)
    {
        $page["title"] = "Team Catalog Detail - TeamModule";

        $data['success'] = true;
        try {
            $team = Team::where("id", $teamID)->with([
                'teamLogs' => function ($logs) {
                    $logs->latest('created_at');
                }
            ])->withTrashed()->first();
            return Inertia::render('Admin/Team/Detail', [
                'team' => $team,
                'teamLogs' => $team->teamLogs,
                'page' => $page,
            ]);
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function updateMetadata(Request $request, $teamID)
    {
        $data['success'] = true;
        try {

            $request['id'] = $teamID;
            $request['updated_by'] = 1;
            $teamService = $this->teamService->updateTeamMetadata($request->all());

            if ($teamService["success"] == true) {

                return Redirect::route('admin.team.detail', $teamID);
            }
        } catch (Exception $ex) {
            Session::flash('success', false);
            Session::flash('message', $ex->getMessage());

            return Redirect::route('admin.team.detail', $teamID);
        }
    }
}
