<?php

namespace App\Http\Controllers\Admin;

use App\Helper\Helper\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Models\PaymentLog;
use App\Models\Role;
use App\Services\RoleService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Spatie\Permission\Models\Role as ModelsRole;
use Spatie\Permission\Models\Permission;

class RoleController extends Controller
{
    private $roleService;

    function __construct(RoleService $roleService)
    {
        $this->roleService = $roleService;
    }

    public function list(Request $request)
    {
        $page["title"] = "Role List - RoleModule";

        $columns = array(
            array("key" => "name", "component" => 'NameColumn', "text" => "Name", "visible" => true, "fixed" => true),
            array("key" => "created_at", "component" => 'CreatedAtColumn', "text" => "Created", "visible" => true),
            array("key" => "updated_at", "component" => 'UpdatedAtColumn', "text" => "Updated", "visible" => true),
        );

        $statuses = ['all', 'available', 'archived'];
       
        $roles = Role::query()
            ->select('id', 'name', 'description', 'color', 'created_by', 'updated_by', 'status', 'created_at', 'updated_at', 'deleted_at')
            ->whereNotIn('name', ['Customer', 'Developer'])
            ->when(!empty($request->status_all), function ($query) use ($request, $statuses) {
                $currentStatus = 0;
                foreach ($statuses as $key => $status) {
                    if ($status == $request->status_all) {
                        $currentStatus = $key;
                    }
                }
                if ($currentStatus == 2) {
                    $query->onlyTrashed();
                } elseif ($currentStatus == 1) {
                    $query->where('status', 1);
                } else {
                    $query->withTrashed();
                }
            })
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        $roleSums = $this->statusCount($statuses);

        return Inertia::render('Admin/Role/List', [
            'columns' => $columns,
            'page' => $page,
            'roles' => collect($roleSums)->merge($roles),
        ]);
    }

    public function statusCount($statuses)
    {
        $statusCounts = [];

        foreach ($statuses as $status) {
            switch ($status) {
                case 'all':
                    // Count products with both statuses 1 and 2
                    $statusCounts['available'] = Role::where('status', 1)->count() + Role::onlyTrashed()->count();
                    break;
                case 'archived':
                    // Count products with status 2
                    $statusCounts['archived'] = Role::onlyTrashed()->count();
                    break;
                case 'available':
                    // Count products with status 1
                    $statusCounts['all'] = Role::where('status', 1)->count();
                    break;
                default:
                    break;
            }
        }

        return $statusCounts;
    }


    
    public function generate(Request $request)
    {
        try {
            $rules =  [
                'name' => 'required',
                'color' => 'required',
            ];

            $CustomMessages = [
                'name.required' => 'Role Name is required.',
                'color.required' => 'Role Color Code is required.',
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.role.list')
                    ->withErrors($validator)
                    ->withInput();
            }

            $request['created_by'] = Auth::id();
            $roleService = $this->roleService->create($request->all());

            if ($roleService["success"] == true) {
                $role_id = $roleService['id'];
                $role_details =  Role::find($role_id);
                $log = new PaymentLog();
                $log->activity = 'A New Role ' . $role_details->name . "'s Created";
                $log->type = 'created';
                $log->request = response()->json($request->all())->content();
                $log->response = response()->json($request->all())->content();
                $log->code = 200;
                $log->created_by = $role_details->created_by;
                $log->loggable()->associate($role_details);
                $log->save();

                NotificationHelper::notify('role', 'A New Role ' . $role_details->name . "'s Created", [
                    'id' => $role_id,
                    'request' => response()->json($request->all())->content(),
                    'response' => response()->json($request->all())->content(),
                    'message' => 'Role Added Successful',
                  ], Auth::user()->id);
                Session::flash('success', true);
                Session::flash('message', 'New Role Created.');
                return Redirect::route('admin.role.list');
            }
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());
            return Redirect::route('admin.role.list');
        }
    }

    public function getRole($roleID)
    {
        $page["title"] = "Edit Role - RoleModule";

        $data['success'] = true;
        try {
            $role = $this->roleService->getRoleById($roleID);

            return response()->json([
                'page' => $page,
                "role" => $role
            ]);
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function update(Request $request, $roleID)
    {
        try {
            $rules = [
                'name' => 'required',
                'color' => 'required',
            ];

            $customMessages = [
                'name.required' => 'Role Name is required.',
                'color.required' => 'Role Color Code is required.',
            ];

            $validator = Validator::make($request->all(), $rules, $customMessages);

            if ($validator->fails()) {
                return Redirect::route('admin.role.detail', $roleID)
                    ->withErrors($validator)
                    ->withInput();
            }
            $role = Role::findOrFail($roleID);
            $currentName = $role->name;
            $roleUpdated = $role->update([
                'name' => $request->input('name'),
                'description' => $request->input('description'),
                'admin_rights' => $request->input('admin_rights') ? 1 : 0,
                'color' => $request->input('color'),
                'updated_by' => Auth::id(),
            ]);

            $assignToRole = ModelsRole::find($role->id);

            if (!empty($request['admin_rights']) && $request['admin_rights']) {
                // Retrieve all permissions associated with the admin role
                $allPermissions = Permission::all();
                $excludedPermissions = [
                    'BlockList-Create', 'BlockList-Delete', 'Refund-View', 'Refund-Create', 
                    'Role-Create', 'Role-Edit', 'Role-View', 'Role-Delete'
                ];

                // Filter out the excluded permissions
                $permissionsToAssign = $allPermissions->filter(function($permission) use ($excludedPermissions) {
                    return !in_array($permission->name, $excludedPermissions);
                });

                // Assign admin permissions to the new role
                $assignToRole->syncPermissions($permissionsToAssign);
            } else if (isset($request['admin_rights']) && !$request['admin_rights']) {
                // Remove all permissions if admin_rights is false
                $assignToRole->syncPermissions([]);

                 $includedPermissions = [
                    'BlockList-View', 'BlockList-Create', 'BlockList-Edit', 
                    'Coupon-View', 'Coupon-Create', 'Coupon-Edit', 
                    'Customer-View', 'Customer-Create', 'Gateway-View', 'Gateway-Create', 
                    'ImportExport-View', 'ImportExport-Create', 
                    'Invoice-View', 'Invoice-Create', 'Invoice-Delete', 
                    'Notes-View', 'Notes-Create', 'Notes-Edit', 
                    'Payment-View', 
                    'PaymentLinkGenerator-View', 'PaymentLinkGenerator-Create', 
                    'Product-View', 
                    'TwoStep-Create', 'TwoStep-Edit'
                ];

                $permissions = Permission::whereIn('name', $includedPermissions)->get();
                $assignToRole->syncPermissions($permissions);
            }
    

            if ($roleUpdated) {
                $log = new PaymentLog();
                $log->activity = 'Role "' . $currentName . '" updated to "' . $role->name  . '"';
                $log->type = 'updated';
                $log->request = response()->json($request->all())->content();
                $log->response = response()->json($request->all())->content();
                $log->code = 200;
                $log->created_by = Auth::id();
                $log->loggable()->associate($role);
                $log->save();
                if($currentName != $role->name){
                    NotificationHelper::notify('role', 'Role "' . $currentName . '" updated to "' . $role->name  . '"', [
                        'id' => $roleID,
                        'request' => response()->json($request->all())->content(),
                        'response' => response()->json($request->all())->content(),
                        'message' => 'Role Updated Successful',
                    ], Auth::user()->id);
                }
                Session::flash('success', true);
                Session::flash('message', 'Role Updated.');
            } else {
                // Flash error message to the session if the update fails
                Session::flash('success', false);
                Session::flash('message', 'Failed to update role.');
            }

            // Flash success message to the session
            Session::flash('success', true);
            Session::flash('message', 'Role Updated.');

            // Redirect to the role detail page
            return Redirect::route('admin.role.detail', $roleID);
        } catch (Exception $ex) {
            // Handle exceptions
            Session::flash('success', false);
            Session::flash('message', $ex->getMessage());

            // Redirect to the role detail page
            return Redirect::route('admin.role.detail', $roleID);
        }
    }

    public function archived($roleID)
    {
        $data['success'] = true;
        try {
            $roleService = $this->roleService->archiveRole($roleID);
            Session::flash('success', true);
            Session::flash('message', $roleService["message"]);
            return Redirect::route('admin.role.list');
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function delete($roleID)
    {
        $data['success'] = true;
        try {
            $roleService = $this->roleService->deleteRole($roleID);

            if ($roleService["success"] == true) {
                Session::flash('success', true);
                Session::flash('message', $roleService["message"]);
                return Redirect::route('admin.role.list');
            }
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function restore($roleID)
    {
        $data['success'] = true;
        try {
            $roleService = $this->roleService->restoreRole($roleID);
            Session::flash('success', true);
            Session::flash('message', $roleService["message"]);
            return Redirect::route('admin.role.list');
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function detail($roleID)
    {
        $page["title"] = "Role Detail - RoleModule";

        $data['success'] = true;
        try {
            $role = Role::where("id", $roleID)->with([
                'roleLogs' => function ($logs) {
                    $logs->latest('created_at');
                }
            ])->withTrashed()->first();
            return Inertia::render('Admin/Role/Detail', [
                'role' => $role,
                'roleLogs' => $role->roleLogs,
                'page' => $page,
            ]);
        } catch (Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function updateMetadata(Request $request, $roleID)
    {
        $data['success'] = true;
        try {

            $request['id'] = $roleID;
            $request['updated_by'] = 1;
            $roleService = $this->roleService->updateRoleMetadata($request->all());

            if ($roleService["success"] == true) {

                return Redirect::route('admin.role.detail', $roleID);
            }
        } catch (Exception $ex) {
            Session::flash('success', false);
            Session::flash('message', $ex->getMessage());

            return Redirect::route('admin.role.detail', $roleID);
        }
    }
}

