<?php

namespace App\Http\Controllers\Admin;

use App\Classes\Refund;
use App\Exceptions\StripeException;
use App\Helper\Helper\StripeHelper;
use App\Http\Controllers\Controller;
use App\Jobs\RefundPayment;
use App\Models\Category;
use App\Models\CountryCurrencies;
use App\Models\CustomerBinding;
use App\Models\Invoice;
use App\Models\Notification;
use App\Models\Payment;
use App\Models\PaymentGateway;
use App\Models\PaymentLink;
use App\Models\RelatedPayments;
use App\Models\Setting;
use App\Models\Team;
use App\Models\User;
use App\Repositories\Payment\Refund\PaypalRefundRepository;
use App\Repositories\Payment\Refund\RefundRepository;
use App\Services\Export\PaymentsExport;
use App\Services\LogService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Spatie\Permission\Models\Role;
use App\Models\Role as RolesModel;

class PaymentController extends Controller
{
    protected $paymentsExport;

    public function __construct(PaymentsExport $paymentsExport)
    {
        $this->paymentsExport = $paymentsExport;
    }

    public function list(Request $request)
    {
        $user = Auth::user();
        $page["title"] = "Payment List - PaymentModule";

        $teams = Team::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();
        $currencies = CountryCurrencies::select('id', 'currency', 'code', 'symbol')->groupBy('code')->get();

        $columns = [
            ['key' => 'price', 'component' => 'AmountColumn', 'text' => 'Amount', 'visible' => true, 'fixed' => true, 'order' => 'desc'],
            ['key' => 'link.item_name', 'component' => 'DescriptionColumn', 'text' => 'Item Name', 'visible' => true, 'order' => 'desc'],
            ['key' => 'customer.email', 'component' => 'CustomerColumn', 'text' => 'Customer', 'visible' => true, 'order' => 'desc'],
            ['key' => 'gateway.name', 'component' => 'MerchantColumn', 'text' => 'Payment Gateway', 'visible' => false, 'order' => 'desc'],
            ['key' => 'card', 'component' => 'CardColumn', 'text' => 'Card', 'visible' => false, 'order' => 'desc'],
            ['key' => 'link.sale_type', 'component' => 'BadgeColumn', 'text' => 'Sales Type', 'visible' => true, 'color' => '#625afa', 'order' => 'desc'],
        ];

        $roles = RolesModel::exceptedRoles(['Super Admin']);

        if ($user->hasRole('Super Admin') || ($user->can('Payment-Global'))) {
            $columns[] = ['key' => 'link.creator.first_name', 'component' => 'BadgeColumn', 'text' => 'Salesperson', 'visible' => true, 'order' => 'desc'];
        }

        $columns[] = ['key' => 'link.team_wise.name', 'component' => 'BadgeColumn', 'text' => 'Team', 'colorKey' => 'link.team_wise.color', 'visible' => false, 'order' => 'desc'];
        $columns[] = ['key' => 'comment', 'component' => 'CommentColumn', 'text' => 'Comment', 'visible' => false, 'order' => 'desc'];
        $columns[] = ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true, 'order' => 'desc'];

        if ($request->order && $request->key) {
            $columns = array_map(function ($column) use ($request) {
                if ($column['key'] === $request->key) {
                    $column['order'] = $request->order;
                }
                return $column;
            }, $columns);
        }

        $statuses = StripeHelper::paymentStatuses();
        $userRoles = $user->roles->pluck('name')->toArray();
        $hasSuperAdminRole = in_array('Super Admin', $userRoles);
        $hasPaymentGlobalPermission = $user->can('Payment-Global');

        $baseQuery = Payment::baseQuery();
        $baseQuery = $baseQuery
            ->when(!empty($request->is_review), function ($query) use ($request) {
                $unReviewedPayments = Notification::selectRaw('json_extract(data, "$.data.id") as id')
                    ->whereNull('read_at')
                    ->where('data->type', 'payment')
                    ->where('notifiable_id', Auth::id())
                    ->get();
                $query->where('status', 1);
                $query->whereIn('id', $unReviewedPayments);

                $unReviewedIds = $unReviewedPayments->pluck('id')->toArray();
                $this->markAllPaymentAsRead($unReviewedIds);
            })
            ->when($hasPaymentGlobalPermission && !$hasSuperAdminRole, function ($query) {
                // Get the IDs of all users with the "Super Admin" role
                $superAdminIds = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Super Admin');
                })->pluck('id');

                $query->whereHas('link', function ($query) use ($superAdminIds) {
                    $query->whereNotIn('created_by', $superAdminIds);
                });
            })
            ->when(!$hasSuperAdminRole && !$hasPaymentGlobalPermission, function ($query) {
                $createdBy = Auth::user()->id;
                $query->whereHas('link', function ($query) use ($createdBy) {
                    $query->where('created_by', $createdBy);
                });
            })
            ->when(!empty($request->status), function ($query) use ($request, $statuses) {
                $mergeStatus = array_intersect($request->status, $statuses);
                $currentStatus = array_map(function ($status) use ($statuses) {
                    return array_search($status, $statuses);
                }, $mergeStatus);
                if (!empty($currentStatus)) {
                    $query->whereIn('status', $currentStatus);
                }
            })
            ->filterByAmount($request, 'amountSearch', 'amountSearchBy')
            ->filterByCurrency($request)
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'] . ":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'] . ":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->filterByCustomerData($request)
            ->filterByPaymentLink($request)
            ->filterByLastFour($request)
            ->filterByCard($request)
            ->when(!empty($request->order), function ($query) use ($request) {
                if (!empty($request->order) && !empty($request->key)) {
                    $query->orderBy($request->key, $request->order);
                } else {
                    $query->orderBy('created_at', $request->order);
                }
            });

        $cloneBaseQuery = clone $baseQuery;

        $baseQuery = $baseQuery->when(!empty($request->status_all), function ($query) use ($request, $statuses) {
            if (in_array($request->status_all, ['fresh_sales', 'upsell'])) {
                $query->whereHas('link', function ($q) use ($request) {
                    $q->where('sale_type', $request->status_all);
                });
            } else {

                $currentStatus = 0;
                foreach ($statuses as $key => $status) {
                    if ($status == $request->status_all) {
                        $currentStatus = $key;
                    }
                }
                if ($currentStatus > 0) {
                    $query->where('status', $currentStatus);
                }
            }
        });

        $payment = (clone $baseQuery)
            ->when(!empty($request->dateSearchBy), function ($query) {
                $query->orderBy('created_at', 'asc');
            }, function ($query) {
                $query->latest('id');
            })
            ->paginate(20)
            ->withQueryString();

        $otherInformation = Payment::getOtherInformation(clone $cloneBaseQuery, $statuses, $request);
        $paymentSums = Payment::queryStatusCount($cloneBaseQuery, $statuses, ($user->hasAnyRole($roles) ? $user->id : ""), $user->roles->pluck('name')[0], $request);

        $gateways = PaymentGateway::select('id', 'name', 'gateway', 'environment', 'created_by')->get();
        $exportColumns = $this->paymentsExport->columns();

        return Inertia::render('Admin/Payment/List', [
            'columns' => $columns,
            'page' => $page,
            'teams' => $teams,
            'categories' => $categories,
            'payments' => $paymentSums->merge($payment),
            'gateways' => $gateways,
            'currencies' => $currencies,
            'exportColumns' => $exportColumns,
            'otherInformation' => $otherInformation,
        ]);
    }

    public function detail(Request $request, $id)
    {
        $paymentDetail = Payment::where("id", $id)->with([
            'link' => function ($link) {
                $link->with('currencyCountry:id,aplha_code3,code,symbol', 'gateway:id,name,gateway,statement_descriptor,environment');
            },
            'customer',
            'paymentLogs' => function ($logs) {
                $logs->orderBy('created_at', 'DESC');
                $logs->orderBy('id', 'DESC');
            }
        ])->first();

        // $relatedPayments = Payment::where("id", "!=", $paymentDetail->id)
        //     ->where("last_four", "!=", 0)
        //     ->where(function ($query) use ($paymentDetail) {
        //         $query->where("ip", $paymentDetail->ip)
        //             ->where("last_four", $paymentDetail->last_four);
        //     })
        //     ->orWhere(function ($query) use ($paymentDetail) {
        //         // Include payments linked via the related_payments table
        //         $query->whereHas('relatedPayments', function ($subQuery) use ($paymentDetail) {
        //             $subQuery->where('customer_id', $paymentDetail->customer_id);
        //         });
        //     })
        //     ->orWhere(function ($query) use ($paymentDetail) {
        //         // Get payments related to the customers manually linked to this payment's customer
        //         $query->whereIn('customer_id', function ($subQuery) use ($paymentDetail) {
        //             $subQuery->select('customer_id')
        //                     ->from('related_payments')
        //                     ->where('payment_id', $paymentDetail->id);
        //         });
        //     })
        //     ->with('customer:id,stripe,email', 'link:id,item_name')
        //     ->whereHas('customer')
        //     ->latest('id')
        //     ->paginate();

        $customer = $paymentDetail->customer;
        $boundCustomerIds = $customer->allBoundCustomers();
        $boundCustomerIds[] = $customer->id; // Include the current customer ID

        $relatedPayments = Payment::whereIn('customer_id', $boundCustomerIds)
            ->where('id', '!=', $paymentDetail->id) // Exclude the current payment
            //->where("last_four", "!=", 0)
            ->orWhere(function ($query) use ($paymentDetail) {
                $query->where("ip", $paymentDetail->ip);
                $query->where("last_four", $paymentDetail->last_four);
                $query->where('id', '!=', $paymentDetail->id);
            })
            ->with('customer:id,stripe,email', 'link:id,item_name')
            ->whereHas('customer', function ($query) {
                $query->whereNull('deleted_at'); // Ensure the payment's customer is not deleted
            })
            ->latest('id')
            ->paginate();

        $this->markOneAsRead($id);

        $invoiceSend = Invoice::where("payment_id", $id)
            ->with([
                'paymentLogs' => function ($logs) {
                    $logs->where('type', '=', 'invoice_send')
                        ->orderBy('created_at', 'ASC')
                        ->orderBy('id', 'ASC');
                }
            ])
            ->get();

        $invoiceHistoryLogs = $invoiceSend->flatMap(function ($invoice) {
            return $invoice->paymentLogs;
        });

        $invoiceHistoryLogs->each(function ($log) {
            $log->decoded_request = json_decode($log->request, true);
        });

        $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');

        return Inertia::render('Admin/Payment/Detail', [
            'itemDetail' => $paymentDetail->link,
            'paymentDetail' => $paymentDetail,
            'paymentLogs' => $paymentDetail->paymentLogs,
            'relatedPayments' => $relatedPayments,
            'invoiceHistory' => $invoiceHistoryLogs,
            'brand_settings' => $brand_settings,
        ]);
    }

    // private function markNotificationsAsRead($paymentId)
    // {
    //     $unReviewedPayments = Notification::selectRaw('json_extract(data, "$.data.id") as id')
    //         ->whereNull('read_at')
    //         ->where('notifiable_id', Auth::id())
    //         ->whereJsonContains('data', ['type' => 'payment'])
    //         ->first();
    //     $json = json_decode($unReviewedPayments->id);
    //     $this->markAsRead($json);
    // }


    public function ipLocation($ip)
    {
        try {
            $ipData = Http::get("http://api.ipstack.com/$ip?access_key=512d57513320595fdaf8d282ce9aeab2");

            if ($ipData->successful()) {
                $data = $ipData->json();
            } else {
                throw new \ErrorException('Third-party is not responding');
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (Exception $ex) {
            return response()->json([
                'success' => false,
                'code' => $ex->getCode(),
                'message' => $ex->getMessage()
            ]);
        }
    }

    public function refund(Request $request)
    {
        $CustomMessages = [
            'payments.required' => 'Payments are required to refund',
        ];

        $validator = Validator::make($request->all(), [
            'payments' => [
                'required',
                'array',
                function ($attribute, $value, $fail) {
                    $existingIds = Payment::whereIn('id', $value)->where('status', 1)->pluck('id', 'price', 'intent_id')->toArray();
                    $missingIds = array_diff($value, $existingIds);

                    if (!empty($missingIds)) {
                        $fail("The following IDs do not exist or their status is not succeeded: " . implode(', ', $missingIds));
                    }
                }
            ],
            'reason' => 'required',
            'detailed_reason' => 'required_if:reason,other'
        ], $CustomMessages);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ]);
        } else {
            $log = app(LogService::class);
            $refundRepo = app(RefundRepository::class);
            $reason = [
                'detailed_reason' => $request->detailed_reason,
                'reason' => $request->reason,
            ];

            foreach ($request->payments as $payment) {
                dispatch(new RefundPayment($payment, $reason, 'refunded', Auth::id(), $refundRepo, $log));
            }

            return response()->json([
                'success' => true,
                'message' => 'Refund is processed successfully.'
            ]);
        }
    }

    public function singleRefund(Request $request)
    {
        $CustomMessages = [
            'payment.required' => 'Payment are required to refund',
        ];

        $validator = Validator::make($request->all(), [
            'payment' => [
                'required',
                'integer',
            ],
            'reason' => 'required',
            'detailed_reason' => 'required_if:reason,other',
            'amount' => [
                'required_if:payment_count,1',
                'numeric'
            ],
            'currency' => 'required',
        ], $CustomMessages);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ]);
        }

        try {
            $log = app(LogService::class);

            if ($request->gateway == 'paypal_creditcard' || $request->gateway == 'paypal') {
                $refundRepo = app(PaypalRefundRepository::class);
            } else {
                $refundRepo = app(RefundRepository::class);
            }

            $reason = [
                'detailed_reason' => $request->detailed_reason,
                'reason' => $request->reason,
                'amount' => $request->amount
            ];

            //$refundRepo->create($request->payment, $reason,$request->amount, 'refunded');

            // For direct request
            //$refund = new Refund($refundRepo, $log);
            //$singleRefund = $refund->singleRefund($request->payment, $reason, $request->amount, 'refunded', Auth::id());

            //foreach ($request->payments as $payment) {
           dispatch(new RefundPayment($request->payment, $reason, $request->amount, 'refunded', Auth::id(), $refundRepo, $log));

             //}
            $payment = Payment::find($request->payment);
           // $payment->status = 10;
            $payment->save();

            return response()->json([
                'success' => true,
                'message' => 'Refund is processed successfully.'
            ]);
        } catch (StripeException $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getError(),
                'message' => $e->getMessage()
            ]);
        } catch (Exception $e) {

            dd($e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function update(Request $request, $paymentID)
    {
        $payment = Payment::where('id', '=', $paymentID)->first();

        if ($request->metadata) {
            $metadata = Arr::pluck($request->metadata, 'value', 'key');

            $intentObj = json_decode($payment->intent);

            if ($intentObj !== null && is_object($intentObj) && property_exists($intentObj, 'metadata')) {
                $intentObj->metadata = $metadata;
            } else {
                $intentObj = (object) ['metadata' => $metadata];
            }

            $payment->update([
                'intent' => json_encode($intentObj)
            ]);
        }

        $form = $request->only('first_name', 'last_name', 'email', 'phone', 'address', 'city', 'company', 'state', 'zipcode', 'country');
        if (!empty($form)) {
            $payment->update($form);
        }

        return Redirect::route('admin.payment.detail.id', ['id' => $payment->id]);
    }

    public function markAllRead()
    {
        Auth::user()->unreadNotifications->markAsRead();
    }

    private function markOneAsRead($notificationIds)
    {
        $unReviewedPayments = Notification::selectRaw('json_extract(data, "$.data.id") as id, read_at')
            ->whereJsonContains('data', ['type' => 'payment'])
            ->whereNull('read_at')
            ->where('notifiable_id', Auth::id())
            ->get();
        $unReviewedIds = $unReviewedPayments->pluck('id')->toArray();
        $commonIds = array_intersect($unReviewedIds, (array) $notificationIds);
        if ($unReviewedPayments) {
            foreach ($commonIds as $commonId) {
                DB::table('notifications')
                    ->whereRaw('json_extract(data, "$.data.id") = ?', [$commonId])
                    ->update(['read_at' => Carbon::now()]);
            }
        } else {
            return Redirect::route('admin.payment.list');
        }
    }

    public function export(Request $request)
    {
        $data['success'] = false;
        $rules = [
            'columns' => 'nullable|array',
            'dateRange' => 'nullable|array',
            'rangeOption' => 'nullable',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $data['errors'] = $validator->errors();
            return response()->json($data, 422);
        }

        try {
            $validatedData = $validator->validated();
            $this->paymentsExport->columns();

            $data['success'] = true;
            $data['data'] = $this->paymentsExport->export($validatedData);
            return response()->json($data);
            //return Excel::download(new UsersExport($validatedData['columns'], $validatedData['dateRange'], $validatedData['rangeOption']), 'customers.xlsx');
        } catch (Exception $ex) {
            // dd($ex);
            $data['message'] = $ex->getMessage();
            return response()->json($data);
        }
    }

    public function viewReceipt(Request $request, $id)
    {
        $data['success'] = true;
        try {
            $payment = Payment::with('invoice')
                ->find($id);

            $invoice = Invoice::with(['payment.link.categories', 'customer'])
                ->find($payment->invoice['id']);

            $data['data'] = $invoice;
        } catch (\Exception $e) {
            $data['message'] = $e->getMessage();
        }
        return response()->json($data);
    }

    public function updateDescription(Request $request)
    {
        $paymentLink = PaymentLink::findOrFail($request->id);

        if (!empty($request->item_name)) {
            $paymentLink->update([
                'item_name' => $request->item_name,
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Description is updated successfully.',
            'description' => $paymentLink->item_name,
        ]);
    }
    public function markAllPaymentAsRead($unReviewedIds)
    {
        $notifications = Notification::where('notifiable_id', Auth::id())->get();

        foreach ($notifications as $notification) {
            $decodedData = json_decode($notification->data);

            if (isset($decodedData->data->id)) {
                if (in_array($decodedData->data->id, $unReviewedIds)) {
                    $notification->read_at = now();
                    $notification->save();
                }
            }
        }
    }

    public function refundPayments(Request $request)
    {
        $rules = [
            'payments' => 'required|array',
            'payments.*' => 'exists:payments,id',
        ];

        $messages = [
            'payments.required' => 'Payments are required to refund',
            'payments.arrat' => 'Payments should be an array',
            'payments.*.exists' => 'The selected payments do not exist',
        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return response()->json(['erros' => $validator->errors()]);
        }

        $payments = Payment::whereIn('id', $request->payments)->with([
            'customer:id,email',
            'link' => function ($link) {
                $link->select('id', 'token', 'valid_till', 'item_name', 'price', 'discount_type', 'discount', 'original_price', 'item_description', 'currency', 'payment_gateway', 'sale_type', 'team', 'created_by');
                $link->with([
                    'currencyCountry:id,code,symbol',
                ]);
            },
        ])
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        return Inertia::render('Admin/Payment/RefundPayments', [
            'payments' => $payments,
            'successCount' => $request->successCount,
            'failedCount' => $request->failedCount,
        ]);
    }

    /**
     * This funciton has to be optimized or maybe define in model
     *
     * Summary of searchPayment
     * @param \Illuminate\Http\Request $request
     * @return mixed|\Illuminate\Http\JsonResponse
     */
    public function searchPayment(Request $request)
    {
        $response = [
            'success' => false,
            'data' => [
                'payments' => [],
                'customers' => [],
                'payment_links' => [],
            ],
            'total' => 0,
            'message' => "",
        ];

        $payments = Payment::with([
            'customer' => function ($query) {
                $query->select('id', 'first_name', 'last_name', 'email');
            },
            'link' => function ($query) {
                $query->select('id', 'item_name', 'price');
            },
        ])
            ->select('id', 'intent_id', 'status', 'price', 'created_at', 'customer_id', 'payment_link_id', DB::raw('sum(price) as sumAmount'))
            ->where(function ($query) use ($request) {
                $query->whereHas('customer', function ($query) use ($request) {
                    $query->where('email', 'like', '%' . $request->keyword . '%')
                        ->orWhere('first_name', 'like', '%' . $request->keyword . '%')
                        ->orWhere('last_name', 'like', '%' . $request->keyword . '%');
                    $query->groupBy('email');
                })
                    ->orWhereHas('link', function ($query) use ($request) {
                        $query->where('price', 'like', '%' . $request->keyword . '%')
                            ->orWhere('item_name', 'like', '%' . $request->keyword . '%');
                        $query->groupBy('id');
                    })
                    ->orWhere('intent_id', 'like', '%' . $request->keyword . '%');
            })
            ->where('status', 1)
            ->groupBy('payment_link_id')
            ->latest('created_at')
            ->get();

        foreach ($payments as $payment) {
            $response['data']['payments'][] = [
                'id' => $payment->id,
                'intentId' => $payment->intent_id,
                'status' => $payment->status,
                'price' => $payment->sumAmount,
                'createdAt' => $payment->created_at->format('Y-m-d'),
            ];

            $response['data']['customers'][] = [
                'id' => $payment->customer?->id,
                'firstName' => $payment->customer?->first_name,
                'lastName' => $payment->customer?->last_name,
                'email' => $payment->customer?->email,
            ];

            $response['data']['payment_links'][] = [
                'id' => $payment->link->id,
                'itemName' => $payment->link->item_name,
            ];
        }

        $response['success'] = true;
        $response['total'] = count($response['data']['payments']);
        $response['message'] = "Payments searched successfully!";

        return response()->json($response);
    }

    function addManuallyPaymentAsRelated(Request $request)
    {
        $data['success'] = false;
        try {
            if (!$request->filled(['customer_id', 'payment_id'])) {
                throw new Exception("Customer/Payment required for relation");
            }

            foreach ($request->payment_id as $payment) {
                RelatedPayments::create([
                    'customer_id' => $request->customer_id,
                    'payment_id' => $payment,
                    'created_by' => Auth::id()
                ]);
            }

            $data['success'] = true;
            $data['message'] = 'Payment successfully related to Customer';
        } catch (Exception $ex) {
            $data['message'] = $ex->getMessage();
        }

        return response()->json($data);
    }

    function bindCustomerManually(Request $request)
    {
        $data['success'] = false;
        try {
            if (!$request->filled(['customer_a_id', 'customer_b_id', 'payment_id'])) {
                throw new Exception("Customer/Payment required for relation");
            }

            foreach ($request->payment_id as $key => $payment) {
                CustomerBinding::create([
                    'customer_a_id' => $request->customer_a_id,
                    'customer_b_id' => $request->customer_b_id[$key],
                    'payment_id' => $payment,
                    'created_by' => Auth::id()
                ]);
            }

            $data['success'] = true;
            $data['message'] = 'Payment successfully related to Customer';
        } catch (Exception $ex) {
            $data['message'] = $ex->getMessage();
        }

        return response()->json($data);
    }
}
