<?php

namespace App\Http\Controllers\Admin;

use App\Classes\SendMail;
use App\Classes\SendTestInvoice;
use App\Helper\Helper\NotificationHelper;
use App\Helper\Helper\StripeHelper;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Payment\InvoiceController as PaymentInvoiceController;
use App\Models\Category;
use App\Models\CountryCurrencies;
use App\Models\EmailTemplate;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\PaymentGateway;
use App\Models\PaymentLink;
use App\Models\PaymentlinkCategory;
use App\Models\Setting;
use App\Models\Team;
use App\Models\User;
use App\Repositories\Payment\Token\TokenRepository;
use App\Services\Export\InvoiceExport;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Dompdf\Dompdf;
use Dompdf\Options;
use Spatie\Permission\Models\Role;
use App\Services\LogService;

class InvoiceController extends Controller
{
    private $token, $paymentlinkCategoryModel, $invoiceExport, $logService;

    public function __construct(TokenRepository $token, PaymentlinkCategory $paymentlinkCategory, InvoiceExport $invoiceExport, LogService $logService) {
        $this->token = $token;
        $this->paymentlinkCategoryModel = $paymentlinkCategory;
        $this->invoiceExport = $invoiceExport;
        $this->logService = $logService;
    }

    public function list(Request $request) {
        $user = Auth::user();
        $page["title"] = "Invoice List - Payment Module";

        $columns = array(
            array("key" => "price", "component" => 'AmountColumn', "text" => "Amount", "visible" => true, "fixed" => true),
            array("key" => "invoice_no", "component" => 'InvoiceColumn', "text" => "Invoice No", "visible" => true),
            array("key" => "customer.email", "component" => 'CustomerColumn', "text" => "Customer", "visible" => true),
            array("key" => "created_at", "component" => 'CreatedAtColumn', "text" => "Created", "visible" => true),
        );

        $getroles = Role::pluck('name')->toArray();

        // Define roles to exclude
        $excludedRoles = ['Super Admin'];

        // Filter out roles excluding the excluded roles
        $filteredRoles = array_diff($getroles, $excludedRoles);

        // If you need to re-index the array after filtering
        $filteredRoles = array_values($filteredRoles);

        $roles = $filteredRoles;

        $statuses = ['all', 'Paid', 'Draft' ];


        $invoicesQuery = Invoice::query()
            ->select('id', 'invoice_no', 'customer_id', 'payment_id', 'email_sent', 'status', 'created_at')
            ->when(!empty($request->status_all), function ($query) use ($request, $statuses) {
                if ($request->status_all != 'all') {
                    $query->where('status',  array_search($request->status_all, $statuses));
                }
            })
            ->when(!empty($request->statusSearchBy), function ($query) use ($request, $statuses) {
                if ($request->statusSearchBy != 'all') {
                    $query->where('status',  array_search($request->statusSearchBy, $statuses));
                }
            })
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'] . ":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'] . ":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }


                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when((!empty($request->amountSearch) && !empty($request->amountSearchBy)), function ($query) use ($request) {
                $query->whereHas('payment', function ($query) use ($request) {
                    $query->where('price', $request->amountSearchBy, $request->amountSearch);
                });
            })
            ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->whereHas('customer', function ($query) use ($request) {
                    $query->where('email', $request->customerEmail);
                });
            })
            ->when(!empty($request->customerPhone), function ($query) use ($request) {
                $query->whereHas('customer', function ($query) use ($request) {
                    $query->where('phone', $request->customerPhone);
                });
            })
            ->with([
                'payment' => function ($payment) {
                    $payment->with(['link' => function ($link) {
                        $link->with('currencyCountry:id,aplha_code3,code,symbol', 'gateway:id,name,gateway,statement_descriptor,environment');
                    }]);
                },
                'customer:id,email,first_name,last_name',
            ]);
        if ($user->hasRole('Super Admin')) {

            $invoices = $invoicesQuery->latest('id')
            ->paginate(20)
            ->withQueryString();

        } elseif($user->hasAnyRole($roles)) {
                if ($user->can('Invoice-Global')) {
                    $superAdminIds = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->pluck('id');

                    $invoices = $invoicesQuery->whereHas('payment', function ($query) use ($superAdminIds) {
                        $query->whereHas('link', function ($innerQuery) use ($superAdminIds) {
                            $innerQuery->whereNotIn('created_by', $superAdminIds);
                        });
                    })->latest('id')
                    ->paginate(20)
                    ->withQueryString();

                } else {
                    $invoices = $invoicesQuery->whereHas('payment', function ($query) use ($user) {
                        $query->whereHas('link', function ($innerQuery) use ($user) {
                            $innerQuery->where('created_by', $user->id);
                        });
                    })->latest('id')
                    ->paginate(20)
                    ->withQueryString();
                }

            }



        $exportColumns = $this->invoiceExport->columns();

        return Inertia::render('Admin/Invoice/List', [
            'exportColumns' => $exportColumns,
            'statuses'=> $statuses,
            'columns' => $columns,
            'invoices' => $invoices,
            'page' => $page,
        ]);

    }

    public function create(Request $request, $customer_id = null) {
        $page["title"] = "Create Invoice - Payment Module";

        $teams = Team::select('id','name')->get();
        $categories = Category::select('id','name')->get();
        $gateways = PaymentGateway::select('id','name')->get();
        $countries = CountryCurrencies::select('id','aplha_code3','currency','code','symbol','country')
        ->orderBy('currency','ASC')
        ->get();

        $customers = User::select('id','first_name','last_name','email','company','phone','address','city','state','zipcode','country')
                        ->role('customer')
                        ->latest('created_at')
                        ->get();

        $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');

        return Inertia::render('Admin/Invoice/Create', [
            'page' => $page,
            'teams' => $teams->map(function($team){
                return [
                    'label' => $team->name,
                    'value' => $team->id
                ];
            }),
            'categories' => $categories,
            'countries' => $countries,
            'gateways' => $gateways->map(function($team){
                return [
                    'label' => $team->name,
                    'value' => $team->id
                ];
            }),
            'customers' => $customers,
            'selected_customer_id' => $customer_id,
            'brand_settings' => $brand_settings,
        ]);
    }

    public function store(Request $request) {
        try {
            $rules =  [
                'payment_link_id' => [
                    'nullable',
                    Rule::exists('payment_links', 'id')->where(function ($query) use ($request) {
                        return $request->filled('payment_link_id');
                    }),
                ],
                'customer_id' => [
                    'nullable',
                    Rule::exists('users', 'id')->where(function ($query) use ($request) {
                        return $request->filled('customer_id');
                    }),
                ],
                'category_id' => [
                    'nullable',
                    Rule::exists('categories', 'id')->where(function ($query) use ($request) {
                        return $request->filled('category_id');
                    }),
                ],
                // Category Data
                'category_id' => 'required_without:category_id|array',
                // Customer Data
                'customer.first_name' => 'required_without:customer_id|string|max:255',
                'customer.last_name' => 'required_without:customer_id|string|max:255',
                'customer.email' => [
                    'required_without:customer_id',
                    'email',
                    'max:255',
                    Rule::unique('users', 'email')->ignore($request->customer_id, 'id')
                ],
                'customer.company' => 'required_without:customer_id|string|max:255',
                'customer.phone' => 'nullable|required_without:customer_id|string|max:255',
                'customer.address' => 'nullable|required_without:customer_id|string|max:255',
                'customer.city' => 'nullable|required_without:customer_id|string|max:255',
                'customer.state' => 'nullable|required_without:customer_id|string|max:255',
                'customer.country' => 'nullable|required_without:customer_id|string|max:255',
                'customer.zipcode' => 'nullable|required_without:customer_id|string|max:255',

                // Payment Data
                'payment.itemName' => 'required_without:payment_link_id|string|max:255',
                'payment.price' => 'required_without:payment_link_id|numeric|min:1',
                'payment.symbol' => 'required_without:payment_link_id|string|max:255',
                'payment.discount' => 'required_without:payment_link_id|numeric',
                'payment.discountType' => 'required_without:payment_link_id|string|max:255',
                'payment.grossTotal' => 'required_without:payment_link_id|numeric|min:0',
                'payment.team' => 'required_without:payment_link_id|numeric|max:255',
                'payment.salesType' => 'required_without:payment_link_id|string|max:255',
                'payment.paymentGateway' => 'required_without:payment_link_id|numeric|max:255',
                'save_draft' => 'nullable'
            ];

            $CustomMessages = [
                'payment.itemName.required_without' => 'The Item Name field is required.',
                'category_id.required_without' => 'The Services field is required.',
                'customer.last_name.required_without' => 'The Last Name field is required.',
                'customer.phone.required_without' => 'The Phone Number field is required.',
                'customer.state.required_without' => 'The State field is required.',
                'customer.email.required_without' => 'The Email field is required.',
                'customer.email.email' => 'The Email must be a valid email address.',
                'customer.email.unique' => 'The Email has already been taken.',
                'customer.company.required_without' => 'The Company field is required.',
                'customer.address.required_without' => 'The Address field is required.',
                'customer.city.required_without' => 'The City field is required.',
                'customer.zipcode.required_without' => 'The Zipcode field is required.',
                'payment.price.min' => 'The Item Price must be greater than 0.',
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $validatorData = $validator->validated();

                // $customer_id = $validatorData['customer_id'];

                // Update existing user if email exists in the table
                $existingUser = User::where('email', $validatorData['customer']['email'])->first();
                if ($existingUser) {
                    $existingUser->update([
                        'first_name' => $validatorData['customer']['first_name'],
                        'last_name' => $validatorData['customer']['last_name'],
                        'company' => $validatorData['customer']['company'],
                        'phone' => $validatorData['customer']['phone'],
                        'address' => $validatorData['customer']['address'],
                        'city' => $validatorData['customer']['city'],
                        'state' => $validatorData['customer']['state'],
                        'country' => $validatorData['customer']['country'],
                        'zipcode' => $validatorData['customer']['zipcode'],
                    ]);

                    $customer_id = $existingUser->id;
                } else {
                // Create a new user
                $customer = User::create([
                    'first_name' => $validatorData['customer']['first_name'],
                    'last_name' => $validatorData['customer']['last_name'],
                    'email' => $validatorData['customer']['email'],
                    'company' => $validatorData['customer']['company'],
                    'password' => Hash::make('12345678'),
                    'phone' => $validatorData['customer']['phone'],
                    'address' => $validatorData['customer']['address'],
                    'city' => $validatorData['customer']['city'],
                    'state' => $validatorData['customer']['state'],
                    'country' => $validatorData['customer']['country'],
                    'zipcode' => $validatorData['customer']['zipcode'],
                    'created_by' => Auth::id()
                ]);

                $customer->assignRole('Customer');
                $customer_id = $customer->id;
            }

            $token['customer_id'] = $customer_id;
            $token['item_name'] = $validatorData['payment']['itemName'];
            $token['original_price'] = $validatorData['payment']['grossTotal'];
            //$discount = $this->discount($validatorData['payment']['price'], $validatorData['payment']['discount'], $validatorData['payment']['discountType']);
            $token['discount'] = $validatorData['payment']['discount'];
            $token['discountType'] = $validatorData['payment']['discountType'];
            $token['item_price'] = $validatorData['payment']['price'];
            $token['saleType'] = $validatorData['payment']['salesType'];
            $token['currency'] = 170;
            $token['team'] = $validatorData['payment']['team'];
            $token['payment_type'] = 'straight';
            $token['paymentGateway'] = $validatorData['payment']['paymentGateway'] ?? 1;
            $token['valid_till'] = Carbon::now()->addHours(48)->timestamp;
            $token['created_by'] = Auth::id();

            $paymentLink = $this->token->generate($token);
            $categoryLinks = $validatorData['category_id'];
            $tokenCategories = $this->token->categories($paymentLink['id'], $categoryLinks, Auth::id());

            $this->paymentlinkCategoryModel->insert($tokenCategories);

            $this->paymentLinkStatus($paymentLink['id'], 2);

            $payment = Payment::create([
                'payment_link_id' => $paymentLink['id'],
                'customer_id' => $customer_id,
                'item_name' => $validatorData['payment']['itemName'],
                'price' => $validatorData['payment']['grossTotal'],
                'discount' => $paymentLink['discount'],
                'currency' => 170,
                'last_four' => 0000,
                'converted_amount'=> $validatorData['payment']['grossTotal'],
                'team' => $validatorData['payment']['team'],
                'sales_type' => $validatorData['payment']['salesType'],
                'status' => (!empty($validatorData['save_draft'])) ? (13) : (15),
            ]);

            $customer = User::where('id', $customer_id)->first();
            NotificationHelper::notify('invoice', 'A new invoice of amount $'.$validatorData['payment']['grossTotal'].' '.'has been generated', [
                'id' => $customer_id,
                'payment_link_id' =>$paymentLink['id'],
                'message' => 'Invoice create successfully'
            ],  Auth::id());
            PaymentInvoiceController::create($payment->id, $validatorData['save_draft']);

            if(empty($validatorData['save_draft'])){
                $sendMail = $this->mailTestInvoice($payment->id, [
                    'to' => $customer->email,
                    'cc' => ''
                ]);

                return response()->json($sendMail);
            } else {

                return response()->json(['success' => true, "message" => "Draft saved."]);
            }

        } catch(Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function update(Request $request, $id) {
        try {
            $rules =  [
                'payment_link_id' => [
                    'nullable',
                    Rule::exists('payment_links', 'id')->where(function ($query) use ($request) {
                        return $request->filled('payment_link_id');
                    }),
                ],
                'customer_id' => [
                    'nullable',
                    Rule::exists('users', 'id')->where(function ($query) use ($request) {
                        return $request->filled('customer_id');
                    }),
                ],
                'category_id' => [
                    'nullable',
                    Rule::exists('categories', 'id')->where(function ($query) use ($request) {
                        return $request->filled('category_id');
                    }),
                ],
                // Category Data
                'category_id' => 'required_without:category_id|array',
                // Customer Data
                'customer.first_name' => 'required_without:customer_id|string|max:255',
                'customer.last_name' => 'required_without:customer_id|string|max:255',
                'customer.email' => [
                    'required_without:customer_id',
                    'email',
                    'max:255',
                    Rule::unique('users', 'email')->ignore($request->customer_id, 'id')
                ],
                'customer.company' => 'required_without:customer_id|string|max:255',
                'customer.phone' => 'nullable|required_without:customer_id|string|max:255',
                'customer.address' => 'nullable|required_without:customer_id|string|max:255',
                'customer.city' => 'nullable|required_without:customer_id|string|max:255',
                'customer.state' => 'nullable|required_without:customer_id|string|max:255',
                'customer.country' => 'nullable|required_without:customer_id|string|max:255',
                'customer.zipcode' => 'nullable|required_without:customer_id|string|max:255',

                // Payment Data
                'payment.itemName' => 'required_without:payment_link_id|string|max:255',
                'payment.price' => 'required_without:payment_link_id|numeric|min:1',
                'payment.symbol' => 'required_without:payment_link_id|string|max:255',
                'payment.discount' => 'required_without:payment_link_id|numeric',
                'payment.discountType' => 'required_without:payment_link_id|string|max:255',
                'payment.grossTotal' => 'required_without:payment_link_id|numeric|min:0',
                'payment.team' => 'required_without:payment_link_id|numeric|max:255',
                'payment.salesType' => 'required_without:payment_link_id|string|max:255',
                'payment.paymentGateway' => 'required_without:payment_link_id|numeric|max:255',
                'save_draft' => 'nullable'
            ];

            $CustomMessages = [
                'payment.itemName.required_without' => 'The Item Name field is required.',
                'category_id.required_without' => 'The Services field is required.',
                'customer.last_name.required_without' => 'The Last Name field is required.',
                'customer.phone.required_without' => 'The Phone Number field is required.',
                'customer.state.required_without' => 'The State field is required.',
                'customer.email.required_without' => 'The Email field is required.',
                'customer.email.email' => 'The Email must be a valid email address.',
                'customer.email.unique' => 'The Email has already been taken.',
                'customer.company.required_without' => 'The Company field is required.',
                'customer.address.required_without' => 'The Address field is required.',
                'customer.city.required_without' => 'The City field is required.',
                'customer.zipcode.required_without' => 'The Zipcode field is required.',
                'payment.price.min' => 'The Item Price must be greater than 0.',
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $validatorData = $validator->validated();

            $invoice = Invoice::with('payment.link.categories')
            ->find($id);

            $paymentID = $invoice->payment->id;
            $paymentLinkID = $invoice->payment->link->id;

            //$customer_id = $validatorData['customer_id'];

                // Update existing user if email exists in the table
                $existingUser = User::where('email', $validatorData['customer']['email'])->first();
                if ($existingUser) {
                    $existingUser->update([
                        'first_name' => $validatorData['customer']['first_name'],
                        'last_name' => $validatorData['customer']['last_name'],
                        'company' => $validatorData['customer']['company'],
                        'phone' => $validatorData['customer']['phone'],
                        'address' => $validatorData['customer']['address'],
                        'city' => $validatorData['customer']['city'],
                        'state' => $validatorData['customer']['state'],
                        'country' => $validatorData['customer']['country'],
                        'zipcode' => $validatorData['customer']['zipcode'],
                    ]);

                    $customer_id = $existingUser->id;
                } else {
                // Create a new user
                $customer = User::create([
                    'first_name' => $validatorData['customer']['first_name'],
                    'last_name' => $validatorData['customer']['last_name'],
                    'email' => $validatorData['customer']['email'],
                    'company' => $validatorData['customer']['company'],
                    'password' => Hash::make('12345678'),
                    'phone' => $validatorData['customer']['phone'],
                    'address' => $validatorData['customer']['address'],
                    'city' => $validatorData['customer']['city'],
                    'state' => $validatorData['customer']['state'],
                    'country' => $validatorData['customer']['country'],
                    'zipcode' => $validatorData['customer']['zipcode'],
                    'created_by' => Auth::id()
                ]);

                $customer->assignRole('Customer');
                $customer_id = $customer->id;
            }

            $token['customer_id'] = $customer_id;
            $token['item_name'] = $validatorData['payment']['itemName'];
            $token['original_price'] = $validatorData['payment']['grossTotal'];
            //$discount = $this->discount($validatorData['payment']['price'], $validatorData['payment']['discount'], $validatorData['payment']['discountType']);
            $token['discount'] = $validatorData['payment']['discount'];
            $token['discountType'] = $validatorData['payment']['discountType'];
            $token['item_price'] = $validatorData['payment']['price'];
            $token['saleType'] = $validatorData['payment']['salesType'];
            $token['currency'] = 170;
            $token['team'] = $validatorData['payment']['team'];
            $token['payment_type'] = 'straight';
            $token['paymentGateway'] = $validatorData['payment']['paymentGateway'] ?? 1;
            $token['valid_till'] = Carbon::now()->addHours(48)->timestamp;
            $token['created_by'] = Auth::id();

            $paymentLink = $this->token->update($token, $paymentLinkID);

            $newCategories = $validatorData['category_id'];

            $existingCategories = PaymentLinkCategory::where('paymentlink_id', $paymentLink['id'])->pluck('category_id')->toArray();
            // Get Categories to Add and Remove base on the updated categories
            $categoryComparison = $this->token->compareCategories($paymentLinkID, $newCategories, $existingCategories);

            $categoriesToAdd = $categoryComparison['categoriesToAdd'];
            $categoriesToRemove = $categoryComparison['categoriesToRemove'];
            // Added New Categories
            $tokenCategories = $this->token->categories($paymentLink['id'], $categoriesToAdd, Auth::id());
            $this->paymentlinkCategoryModel->insert($tokenCategories);
            // Remove Old Categories that are not comming in updated record
            $this->token->deleteCategoriesToRemove($paymentLink['id'], $categoriesToRemove);


            $payment = Payment::findOrFail($paymentID);

            $payment->update([
                'payment_link_id' => $paymentLink['id'],
                'customer_id' => $customer_id,
                'item_name' => $validatorData['payment']['itemName'],
                'price' => $validatorData['payment']['grossTotal'],
                'discount' => $paymentLink['discount'],
                'currency' => 170,
                'last_four' => 0000,
                'converted_amount'=> $validatorData['payment']['grossTotal'],
                'team' => $validatorData['payment']['team'],
                'sales_type' => $validatorData['payment']['salesType'],
                'status' => (!empty($validatorData['save_draft'])) ? (13) : (15),
            ]);

            $customer = User::where('id', $customer_id)->first();
            PaymentInvoiceController::update($id, $payment->id, $validatorData['save_draft']);

            if(empty($validatorData['save_draft'])){
                return response()->json(['success' => true, "message" => "Invoice Updated Succesfully."]);
            } else {

                return response()->json(['success' => true, "message" => "Draft Updated."]);
            }

        } catch(Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }

    }

    public function detail(Request $request, $id) {
        $page["title"] = "Detailed Invoice - Payment Module";

        $invoice = Invoice::where("id", $id)->with([
            'payment' => function ($payment) {
                $payment->select('id','customer_id','payment_link_id','price','discount','currency','status','created_at');
                $payment->with([
                    'customer:id,first_name,last_name,company,email,phone,city,state,country,zipcode',
                    'link'=> function($link) {
                        $link->with('currencyCountry:id,currency,aplha_code3,code,symbol', 'gateway:id,name,gateway,statement_descriptor,environment');
                        $link->with('categories:id,name');
                    },
                ]);

            },'paymentLogs' => function($logs) {
                $logs->latest('created_at');
            },
            ])->firstOrFail();

        $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');

        return Inertia::render('Admin/Invoice/Detail', [
            'page' => $page,
            'invoice' => $invoice,
            'brand_settings' => $brand_settings,
        ]);
    }

    public function updateMetadata(Request $request, $id) {
        $invoice = Invoice::where('id', '=', $id)->first();

        if($request->metadata) {
            $metadata = Arr::pluck($request->metadata, 'value', 'key');

            $invoiceObj = json_decode($invoice->data);

            if($invoiceObj?->metadata) {
                $invoiceObj->metadata = $metadata;
            } else {
                $invoiceObj['metadata'] = $metadata;
            }

            $invoice->update([
                'data' => json_encode($invoiceObj)
            ]);
        }

        return Redirect::route('admin.invoice.detail.id', [ 'id' => $invoice->id ]);
    }

    public function downloadInvoice(Request $request, $id)
    {
        $payment_info = Payment::select('id','customer_id','payment_link_id','price','discount','currency','ip','comment','status','created_at')
            ->where('id', $id)
            ->with([
                'currencyCountry:id,code,symbol',
                'link' => function($link) {
                    $link->select('id','token','customer_id','item_name','price','discount','discount_type','item_description','currency','status','created_at');
                    $link->with('categories:id,name');
                }
            ])->first()->toArray();

        $brandSettings['settings'] = (new PaymentInvoiceController)->get_setting()->toArray();
        $user_invoice = Invoice::select('id','invoice_no','custom_id','customer_id','payment_id','email_sent','created_by','status','created_at')
                        ->where('payment_id', $id)->latest('id')->first()->toArray();
        $customer_info = User::select('id','first_name','last_name','email','phone','city','address','state','zipcode','country','company')
                        ->where('id', $payment_info['customer_id'])->first()->toArray();
        $category_info = $payment_info['link']['categories'];
        $currency_info = $payment_info['currency_country'];
        $createdAt = Carbon::parse($user_invoice['created_at']);
        $invoice_date = array('invoice_date' => $createdAt->format('d M Y') );
        $invoice_info = array_merge($brandSettings, $user_invoice, $customer_info, $payment_info,$payment_info['link'], $category_info, $invoice_date, $currency_info);

        try {
            $emailTemplate = EmailTemplate::where('name','Billing')->first();
            $isGeneratingPDF = true;
            $invoice_info['downloadPDF'] = $isGeneratingPDF;
            $parseHTML = Blade::render($emailTemplate->content, $invoice_info);

            $options = new Options();
            $options->set('isHtml5ParserEnabled', true);
            $options->set('isPhpEnabled', true);
            $dompdf = new Dompdf($options);

            $dompdf->loadHtml($parseHTML);
            $dompdf->getOptions()->setDefaultFont('Helvetica');
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();
            $imageData = $dompdf->output();

            $imagePdfFileName = $user_invoice['invoice_no'] . '_image.pdf';
            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $imagePdfFileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch(Exception $e) {
          return  $e->getMessage();
        }
    }

    public function sendTestingInvoice(Request $request) {
        $emails = array(
            'to' => $request->emails,
            'cc' => $request->ccEmails
        );

        $sendMail = $this->mailTestInvoice($request->payment_id, $emails);

        return response()->json($sendMail);
    }

    public function duplicateInvoice(Request $request) {
        try {
            $rules =  [
                'invoice_id' => 'required|exists:invoices,id',
            ];

            $CustomMessages = [
                'invoice_id.required' => 'Item Name is required.',
            ];

            $validator = Validator::make( $request->all(), $rules, $CustomMessages);

            if($validator->fails()) {
                return Redirect::route('admin.payment.generate')
                        ->withErrors($validator)
                        ->withInput();
            }

            $invoice = Invoice::find($request->invoice_id);
            $duplicateInvoice = $invoice->replicate();
            $duplicateInvoice->custom_id = StripeHelper::generateUniqueID('in', 10);

            // Invoice No
            $now = Carbon::now();
            $invoice_date = $now->year . $now->month . $now->day;
            $last_inv = Invoice::latest('id')->first();

            if (!empty($last_inv)) {
                $invoice_no_str = explode("-", $last_inv->invoice_no);
                $last_id = $invoice_no_str[2];
                $invoice_no = "INV-" . $invoice_date . "-" . ($last_id + 1);
            } else {
                $invoice_no = "INV-" . $invoice_date . "-" . ($last_inv + 1 + 1000);
            }

            $duplicateInvoice->invoice_no = $invoice_no;
            $duplicateInvoice->email_sent = 0;
            $duplicateInvoice->status = 2;
            $duplicateInvoice->save();

            $data['status'] = true;
            $data['data']['id'] = $duplicateInvoice->id;
            $data['data']['invoiceNo'] = $duplicateInvoice->invoice_no;
        } catch(Exception $e) {
            $data['status'] = false;
            $data['message'] = $e->getMessage();
        }

        return response()->json($data);
    }

    public function mailTestInvoice($paymentID, $emails) {
        $payment_info = Payment::select('id','customer_id','payment_link_id','price','discount','currency','ip','comment','status','created_at')
            ->where('id', $paymentID)
            ->with([
                'currencyCountry:id,code,symbol',
                'link' => function($link) {
                    $link->select('id','token','customer_id','item_name','price','discount','discount_type','item_description','currency','status','created_at');
                    $link->with('categories:id,name');
                }
            ])->first()->toArray();

        $brandSettings['settings'] = (new PaymentInvoiceController)->get_setting()->toArray();
        $user_invoice = Invoice::select('id','invoice_no','custom_id','customer_id','payment_id','email_sent','created_by','status','created_at')
                        ->where('payment_id', $paymentID)->latest('id')->first()->toArray();
        $customer_info = User::select('id','first_name','last_name','email','phone','city','address','state','zipcode','country','company')
                        ->where('id', $payment_info['customer_id'])->first()->toArray();
        $category_info = $payment_info['link']['categories'];
        $currency_info = $payment_info['currency_country'];
        $createdAt = Carbon::parse($user_invoice['created_at']);
        $invoice_date = array('invoice_date' => $createdAt->format('d M Y') );
        $invoice_info = array_merge($brandSettings, $user_invoice, $customer_info, $payment_info, $category_info, $invoice_date, $currency_info);

        $customer = User::find($customer_info['id']);

        if (!empty($invoice_info)) {
            if (!empty($customer_info)) {
                $sendMail = new SendTestInvoice($invoice_info, $customer, "Billing", ['emails'=> $emails['to'], 'ccEmails'=> $emails['cc']]);
                $sendMail->generate();
                $sendMail->mail();

                /*if (empty($user_invoice["email_sent"])) {
                    return response()->json(['status' => false, "message" => "Invoice not sent."]);
                }*/

                $this->logService->log('invoice', [
                    'activity' => $customer_info['email']."'s invoice for ". $payment_info['currency_country']['symbol'] . $payment_info['price'].' '. $payment_info['currency_country']['code'].' was sent to '.$emails['to'],
                    'loggable_id' => $user_invoice['id'],
                    'type' => 'invoice_send',
                    'request' => response()->json($invoice_info)->content(),
                    'response' => response()->json($invoice_info)->content(),
                    'created_by' => Auth::user()->id,
                ]);

                return ['success' => true, "message" => "Invoice sent."];
            } else {
                return ['success' => false, "message" => "Customer not exist."];
            }
        } else {
            return ['success' => false, "message" => "No Data"];
        }
    }

    public function getInvoice($id) {
        $page["title"] = "Edit Invoice - Payment Module";

        $invoice = Invoice::with('payment.link.categories')
                    ->find($id);

        $teams = Team::select('id','name')->get();
        $categories = Category::select('id','name')->get();
        $gateways = PaymentGateway::select('id','name')->get();
        $countries = CountryCurrencies::select('id','aplha_code3','currency','code','symbol','country')
        ->orderBy('currency','ASC')
        ->get();

        $customers = User::select('id','first_name','last_name','email','company','phone','address','city','state','zipcode','country')
                        ->role('customer')
                        ->latest('created_at')
                        ->get();

        $brand_settings = Arr::pluck(Setting::get(), 'value', 'key');

        return Inertia::render('Admin/Invoice/Edit', [
            'page' => $page,
            'invoice' => $invoice,
            'teams' => $teams->map(function($team){
                return [
                    'label' => $team->name,
                    'value' => $team->id
                ];
            }),
            'categories' => $categories,
            'countries' => $countries,
            'gateways' => $gateways->map(function($team){
                return [
                    'label' => $team->name,
                    'value' => $team->id
                ];
            }),
            'customers' => $customers,
            'brand_settings' => $brand_settings,
        ]);
    }

    public function delete($invoiceID){
        try {
            $invoice = Invoice::withTrashed()->find($invoiceID);
            if (!empty($invoice)) {
                $invoice->forceDelete(); // Permanent delete
                $data['success'] = true;
                $data['message'] = 'Invoice Draft Deleted Successfully';

                Session::flash('success', true);
                Session::flash('message', $data['message']);
                return Redirect::route('admin.invoice.list');
            }

        } catch(Exception $ex) {
            return response()->json([
                "message" => $ex->getMessage()
            ]);
        }
    }

    public function export(Request $request) {
        $data['success'] = false;
        $rules = [
            'columns' => 'nullable|array',
            'dateRange' => 'nullable|array',
            'rangeOption' => 'nullable',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $data['errors'] = $validator->errors();
            return response()->json($data, 422);
        }

        try {
            $validatedData = $validator->validated();
            $this->invoiceExport->columns();

            $data['success'] = true;
            $data['data'] = $this->invoiceExport->export($validatedData);
            return response()->json($data);
        } catch (Exception $ex) {
            $data['message'] = $ex->getMessage();
            return response()->json($data, $ex->getCode());
        }

    }

    protected function paymentLinkStatus($paymentLink_id, $status)
    {
        $paymentLink = PaymentLink::find($paymentLink_id);
        $paymentLink->status = $status;

        if ($paymentLink->save()) {
            return true;
        } else {
            return false;
        }
    }
}
