<?php

namespace App\Http\Controllers\Admin;

use App\Helper\DashboardHelper;
use App\Helper\Helper\StripeHelper;
use App\Http\Controllers\Controller;
use App\Models\CountryCurrencies;
use App\Models\Invoice;
use App\Models\Notification;
use App\Models\Payment;
use App\Models\PaymentLink;
use App\Models\User;
use App\Models\Role as RolesModel;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use App\Models\Brief;
use App\Models\MobileAppBrief;
use App\Models\VideoBrief;
use App\Models\WebBrief;
use App\Models\SmmBrief;
use App\Models\SemBrief;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $roles = RolesModel::exceptedRoles(['Super Admin']);

        $selectedCurrency = $request->input('selectedCurrency', 'USD');
        $currencies = CountryCurrencies::select('id', 'aplha_code3', 'currency', 'code', 'symbol', 'country')
            ->where('code', $selectedCurrency)
            ->groupBy('code')
            ->orderBy('id')
            ->get();

        if ($currencies->isNotEmpty()) {
            $currencyId = $currencies->first()->id;
        } else {
            $currencyId = 170;
        }

        $balanceAmount = 0;
        $monthlyBalance = 0;
        $yearlyBalance = 0;
        $monthlyGrossBalance = 0;

        $date = Carbon::now();
        $user = Auth::user();
        $currentHour = $date->format('h');
        $page["title"] = "Dashboard - PaymentModule";
        $page["page_updated_at"] = $date->format('g:i A');

        if ($request->dateDuration) {
            if ($request->dateDuration == 'All time') {
                $paymentDates = Payment::selectRaw('MIN(created_at) as startDate, MAX(created_at) as endDate')->first();

                if ($paymentDates?->startDate && $paymentDates?->endDate) {
                    $request->merge([
                        'startDate' => Carbon::createFromFormat('Y-m-d H:i:s', $paymentDates->startDate)->format('Y-m-d'),
                        'endDate' => Carbon::createFromFormat('Y-m-d H:i:s', $paymentDates->endDate)->format('Y-m-d'),
                    ]);
                }
            } else {
                $duration = DashboardHelper::getDatesByDuration($request->dateDuration);
                $request->merge([
                    'startDate' => $duration['startDate']->format('Y-m-d'),
                    'endDate' => $duration['endDate']->format('Y-m-d'),
                ]);
            }
        }

        $startDate = $request->startDate ?? now()->subDays(7)->format('Y-m-d');
        $endDate = $request->endDate ?? now()->format('Y-m-d');

        if ($user->hasRole('Super Admin')) {
            if (!empty($request->selectedDate)) {
                $selectedDate['date'] = $request->selectedDate;
                $selectedDate['total'] =  Payment::select(DB::raw('COUNT(status) as total'), DB::raw('DATE(created_at) as date'))->where('status', 1)->whereDate('created_at', '=', $request->selectedDate)->groupBy('date')->first();
            } else {
                $lastPaymentDate = Payment::select(DB::raw('COUNT(status) as total'), DB::raw('DATE(created_at) as date'))->where('status', 1)->whereDate('created_at', '!=', Carbon::now()->format('Y-m-d'))->groupBy('date')->latest('created_at')->first();
                $selectedDate['date'] = $lastPaymentDate->date ?? 0;
                $selectedDate['total'] = $lastPaymentDate->total ?? 0;
            }
            $paymentsBar = $this->paymentsCountFilter($startDate, $endDate);

            $freshsalesCount = Payment::getSaleTypeCountFilter("fresh_sales", $startDate, $endDate);
            $upsellCount = Payment::getSaleTypeCountFilter("upsell", $startDate, $endDate);

            $failedPayments = Payment::lastPayments('failed', $startDate, $endDate)->limit(3)->get();
            $topCustomerBySpend = User::topSpendCustomers($startDate, $endDate);
            $grossVolume = Payment::select(DB::raw('SUM(price) as total'), DB::raw('DATE(created_at) as date'))->whereDate('created_at', '>=', $startDate)->whereDate('created_at', '<=', $endDate)->groupBy('date')->get();
            $netVolumeFromSales = $grossVolume;
            $newCustomers = User::select(DB::raw('COUNT(id) as total'), DB::raw('DATE(created_at) as date'))->role('customer')->whereDate('created_at', '>=', $startDate)->whereDate('created_at', '<=', $endDate)->groupBy('date')->get();
            $highRiskPayments = $this->getHighRiskPayments($startDate, $endDate);
            $customersData = User::select('id', 'first_name', 'last_name', 'email', 'created_at')->role('customer')->withSum('payments', 'price')->whereHas('payments', function ($query) use ($startDate, $endDate) {
                $query->whereDate('created_at', '>=', $startDate);
                $query->whereDate('created_at', '<=', $endDate);
            })->whereDate('created_at', '>=', $startDate)->whereDate('created_at', '<=', $endDate)->latest('created_at')->limit(3)->get();
            $successfulPayments = Payment::select(
                DB::raw('COUNT(status) as total'),
                DB::raw('DATE(created_at) as date')
            )
                ->where('status', 1)
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->groupBy('date')
                ->get();

                $singleDay['NewSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'fresh_sales');
                $singleDay['NewSale']['today'] = $this->getSales(today(), 'fresh_sales');

                $singleDay['UpSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'upsell');
                $singleDay['UpSale']['today'] = $this->getSales(today(), 'upsell');

            $spendPerCustomer = Payment::select(DB::raw('AVG(price) as total'), DB::raw('DATE(created_at) as date'))->where('status', 1)->whereDate('created_at', '>=', $startDate)->whereDate('created_at', '<=', $endDate)->groupBy('date')->get();

            $upSalesGraph = $this->getSaleByTypeAndDate('upsell', $startDate, $endDate);
            $freshSalesGraph = $this->getSaleByTypeAndDate('fresh_sales', $startDate, $endDate);

            $singleDay['GrossVolumes']['lastPayment'] = Payment::select(DB::raw('SUM(price) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', $selectedDate['date'])->groupBy('hour')->get();
            $singleDay['NewCustomers']['lastPayment'] = User::select(DB::raw('COUNT(id) as total'), DB::raw('HOUR(created_at) as hour'))->role('customer')->whereDate('created_at', '=', $selectedDate['date'])->groupBy('hour')->get();
            $singleDay['SuccessfulPayments']['lastPayment'] = Payment::select(DB::raw('COUNT(status) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', $selectedDate['date'])->groupBy('hour')->get();
            $singleDay['GrossVolumes']['today'] = Payment::select(DB::raw('SUM(price) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))->groupBy('hour')->get();
            $singleDay['NewCustomers']['today'] = User::select(DB::raw('COUNT(id) as total'), DB::raw('HOUR(created_at) as hour'))->role('customer')->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))->groupBy('hour')->get();
            $singleDay['SuccessfulPayments']['today'] = Payment::select(DB::raw('COUNT(status) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))->groupBy('hour')->get();

            $balanceAmount = Payment::select(DB::raw('SUM(price) as total'))->where('currency', $currencyId)->where('status', 1)->first();
            $monthlyBalance = Payment::select(DB::raw('SUM(price) as total'))->whereDate('created_at', '>=', Carbon::now()->firstOfMonth()->format('Y-m-d'))->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))->where('currency', $currencyId)->where('status', 1)->first();
            $yearlyBalance = Payment::select(DB::raw('SUM(price) as total'))->whereDate('created_at', '>=', Carbon::now()->firstOfYear()->format('Y-m-d'))->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))->where('currency', $currencyId)->where('status', 1)->first();
            $dailyUpSell = $this->getSaleByTypeDateAndCurrency('upsell', Carbon::now()->startOfDay(), Carbon::now()->endOfDay(), $currencyId);
            $dailyFreshSell = $this->getSaleByTypeDateAndCurrency('fresh_sales', Carbon::now()->startOfDay(), Carbon::now()->endOfDay(), $currencyId);

            $monthlyUpSell = $this->getSaleByTypeDateAndCurrency('upsell', Carbon::now()->firstOfMonth(), Carbon::now(), $currencyId);
            $monthlyFreshSell = $this->getSaleByTypeDateAndCurrency('fresh_sales', Carbon::now()->firstOfMonth(), Carbon::now(), $currencyId);


            $monthlyGrossBalance = Payment::select(DB::raw('SUM(price) as total'))
            ->whereDate('created_at', '>=', Carbon::now()->firstOfMonth()->format('Y-m-d'))
            ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
            ->where('currency', $currencyId)
            ->whereIn('status', [1, 11])
            ->first();

            $invoices = Invoice::select('id', 'invoice_no', 'custom_id', 'customer_id', 'payment_id', 'created_by', 'status', 'created_at')
                ->with([
                    'paymentLink' => function ($query) {
                        $query->select('payment_links.id', 'payment_links.price', 'payment_links.currency')
                            ->with([
                                'currencyCountry:id,symbol'
                            ]);
                    },
                    'customer:id,email'
                ])
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->orderBy('id', 'DESC')
                ->limit(3)
                ->get();

            $refundPayments = Payment::refundCount($startDate, $endDate);

            $refundPaymentsData = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'intent_id', 'created_at')
                ->with(['customer:id,email', 'currencyCountry:id,symbol'])
                ->whereHas('customer', function ($query) {
                    $query->where('deleted_at', null);
                })
                ->whereIn('status', [9, 10, 11])
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->orderByDesc('id')->limit(4)->get();
        } else if ($user->hasAnyRole($roles)) {
            if ($user->can('Payment-Global')) {

                $superAdminIds = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Super Admin');
                })->pluck('id');

                if (!empty($request->selectedDate)) {
                    $selectedDate['date'] = $request->selectedDate;
                    $selectedDate['total'] = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date')
                    )
                        ->whereHas('link', function ($query) use ($superAdminIds) {
                            $query->whereNotIn('created_by', $superAdminIds);
                        })
                        ->where('status', 1)
                        // ->where('customer_id', '=', $user->id)
                        ->whereDate('created_at', '=', $request->selectedDate)
                        ->groupBy('date')
                        ->first();
                } else {
                    $lastPaymentDate = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date'),
                    )
                        ->whereHas('link', function ($query) use ($superAdminIds) {
                            $query->whereNotIn('created_by', $superAdminIds);
                        })
                        ->where('status', 1)
                        ->whereDate('created_at', '!=', Carbon::now()->format('Y-m-d'))
                        ->orderBy('created_at', 'desc')
                        ->groupBy('date')
                        ->first();

                    $selectedDate['date'] = $lastPaymentDate->date ?? 0;
                    $selectedDate['total'] = $lastPaymentDate->total ?? 0;
                }
                $paymentsBar = $this->paymentsCountFilterById($startDate, $endDate, false);

                $freshsalesCount = Payment::getSaleTypeCountFilter("fresh_sales", $startDate, $endDate, $superAdminIds, false);
                $upsellCount = Payment::getSaleTypeCountFilter("upsell", $startDate, $endDate, $superAdminIds, false);

                $failedPayments = Payment::lastPayments('failed', $startDate, $endDate)
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->limit(3)
                    ->get();

                $topCustomerBySpend = User::topSpendCustomers(startDate: "", endDate: "", userIDs: $superAdminIds);
                $grossVolume = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->whereBetween('created_at', [$startDate, $endDate])
                    // ->whereDate('created_at', '>=', $startDate)
                    // ->whereDate('created_at', '<=', $endDate)
                    ->groupBy('date')
                    ->get();


                $netVolumeFromSales = $grossVolume;
                $newCustomers = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                    ->role('customer')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    // ->whereDate('created_at', '>=', $startDate)
                    // ->whereDate('created_at', '<=', $endDate)
                    ->groupBy('date')
                    ->get();

                $highRiskPayments = $this->getHighRiskPayments($startDate, $endDate);

                $customersData = User::select('id', 'first_name', 'last_name', 'email', 'created_at')
                    ->role('customer')
                    ->withSum('payments', 'price')
                    ->whereHas('payments', function ($query) use ($startDate, $endDate) {
                        $query->whereDate('created_at', '>=', $startDate);
                        $query->whereDate('created_at', '<=', $endDate);
                    })
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->latest('created_at')
                    ->limit(3)
                    ->get();
                $successfulPayments = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->groupBy('date')->get();
                $spendPerCustomer = Payment::select(
                    DB::raw('AVG(price) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                ->whereHas('link', function ($query) use ($superAdminIds) {
                    $query->whereNotIn('created_by', $superAdminIds);
                })
                ->where('status', 1)
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->groupBy('date')
                ->get();

            $upSalesGraph = $this->getSaleByTypeAndDate('upsell', $startDate, $endDate);
            $freshSalesGraph = $this->getSaleByTypeAndDate('fresh_sales', $startDate, $endDate);

                    $singleDay['NewSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'fresh_sales');
                    $singleDay['NewSale']['today'] = $this->getSales(today(), 'fresh_sales');

                    $singleDay['UpSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'upsell');
                    $singleDay['UpSale']['today'] = $this->getSales(today(), 'upsell');

                $singleDay['GrossVolumes']['lastPayment'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();
                $singleDay['NewCustomers']['lastPayment'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['lastPayment'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();

                $singleDay['GrossVolumes']['today'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['NewCustomers']['today'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['today'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $balanceAmount = Payment::with('link')
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('currency', $currencyId)
                    ->where('status', 1)
                    ->select(DB::raw('SUM(price) as total'))
                    ->first();
                $monthlyBalance = Payment::select(DB::raw('SUM(price) as total'))
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->whereDate('created_at', '>=', Carbon::now()->firstOfMonth()->format('Y-m-d'))
                    ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
                    ->where('currency', $currencyId)
                    ->where('status', 1)->first();

                $yearlyBalance = Payment::select(DB::raw('SUM(price) as total'))
                ->whereHas('link', function ($query) use ($superAdminIds) {
                    $query->whereNotIn('created_by', $superAdminIds);
                })
                ->whereDate('created_at', '>=', Carbon::now()->firstOfYear()->format('Y-m-d'))
                ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
                ->where('currency', $currencyId)
                ->where('status', 1)->first();

                $dailyUpSell = $this->getSaleByTypeDateAndCurrency('upsell', Carbon::now()->startOfDay(), Carbon::now()->endOfDay(), $currencyId, $superAdminIds);
                $dailyFreshSell = $this->getSaleByTypeDateAndCurrency('fresh_sales', Carbon::now()->startOfDay(), Carbon::now()->endOfDay(), $currencyId, $superAdminIds);

                $monthlyUpSell = $this->getSaleByTypeDateAndCurrency('upsell', Carbon::now()->firstOfMonth(), Carbon::now(), $currencyId, $superAdminIds);
                $monthlyFreshSell = $this->getSaleByTypeDateAndCurrency('fresh_sales', Carbon::now()->firstOfMonth(), Carbon::now(), $currencyId, $superAdminIds);

                $monthlyGrossBalance = Payment::select(DB::raw('SUM(price) as total'))
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->whereDate('created_at', '>=', Carbon::now()->firstOfMonth()->format('Y-m-d'))
                    ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
                    ->where('currency', $currencyId)
                    ->whereIn('status', [1, 11])
                    ->first();

                $invoices = Invoice::select('id', 'invoice_no', 'custom_id', 'customer_id', 'payment_id', 'created_by', 'status', 'created_at')
                    ->with([
                        'paymentLink' => function ($query) {
                            $query->select('payment_links.id', 'payment_links.price', 'payment_links.currency')
                                ->with([
                                    'currencyCountry:id,symbol'
                                ]);
                        },
                        'customer:id,email'
                    ])
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->whereNotIn('created_by', $superAdminIds)
                    ->orderBy('id', 'DESC')
                    ->limit(3)
                    ->get();

                $refundPayments = Payment::refundCount($startDate, $endDate, $superAdminIds);

                $refundPaymentsData = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'intent_id', 'created_at')
                    ->with('customer:id,email')
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->whereHas('customer', function ($query) {
                        $query->where('deleted_at', null);
                    })
                    ->whereIn('status', [9, 10, 11])
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->orderByDesc('id')->limit(4)->get();
            } else {

                if (!empty($request->selectedDate)) {
                    $selectedDate['date'] = $request->selectedDate;
                    $selectedDate['total'] = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date')
                    )
                        ->whereHas('link', function ($query) use ($user) {
                            $query->where('created_by', $user->id);
                        })
                        ->where('status', 1)
                        // ->where('customer_id', '=', $user->id)
                        ->whereDate('created_at', '=', $request->selectedDate)
                        ->groupBy('date')
                        ->first();
                } else {
                    $lastPaymentDate = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date'),
                    )
                        ->whereHas('link', function ($query) use ($user) {
                            $query->where('created_by', $user->id);
                        })
                        ->where('status', 1)
                        ->whereDate('created_at', '!=', Carbon::now()->format('Y-m-d'))
                        ->orderBy('created_at', 'desc')
                        ->groupBy('date')
                        ->first();

                    $selectedDate['date'] = $lastPaymentDate->date ?? 0;
                    $selectedDate['total'] = $lastPaymentDate->total ?? 0;
                }
                $paymentsBar = $this->paymentsCountFilterById($startDate, $endDate);

                $freshsalesCount = Payment::getSaleTypeCountFilter("fresh_sales", $startDate, $endDate, $user->id);
                $upsellCount = Payment::getSaleTypeCountFilter("upsell", $startDate, $endDate, $user->id);

                $failedPayments = Payment::lastPayments('failed', $startDate, $endDate)
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->limit(3)
                    ->get();
                $topCustomerBySpend = User::topSpendCustomers($startDate, $endDate, $user->id, true);
                $grossVolume = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->whereBetween('created_at', [$startDate, $endDate])
                    // ->whereDate('created_at', '>=', $startDate)
                    // ->whereDate('created_at', '<=', $endDate)
                    ->groupBy('date')
                    ->get();


                $netVolumeFromSales = $grossVolume;
                $newCustomers = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                    ->role('customer')
                    ->whereBetween('created_at', [$startDate, $endDate])
                    // ->whereDate('created_at', '>=', $startDate)
                    // ->whereDate('created_at', '<=', $endDate)
                    ->groupBy('date')
                    ->get();

                $highRiskPayments = $this->getHighRiskPayments($startDate, $endDate);

                $customersData = User::select('id', 'first_name', 'last_name', 'email', 'created_at')
                    ->role('customer')
                    ->withSum('payments', 'price')
                    ->whereHas('payments', function ($query) use ($startDate, $endDate) {
                        $query->whereDate('created_at', '>=', $startDate);
                        $query->whereDate('created_at', '<=', $endDate);
                    })
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->latest('created_at')
                    ->limit(3)
                    ->get();
                $successfulPayments = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->groupBy('date')->get();
                $spendPerCustomer = Payment::select(
                    DB::raw('AVG(price) as total'),
                    DB::raw('DATE(created_at) as date')
                )
                ->whereHas('link', function ($query) use ($user) {
                    $query->where('created_by', $user->id);
                })
                ->where('status', 1)
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->groupBy('date')
                ->get();

            $upSalesGraph = $this->getSaleByTypeAndDate('upsell', $startDate, $endDate);
            $freshSalesGraph = $this->getSaleByTypeAndDate('fresh_sales', $startDate, $endDate);

                    $singleDay['NewSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'fresh_sales');
                    $singleDay['NewSale']['today'] = $this->getSales(today(), 'fresh_sales');

                    $singleDay['UpSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'upsell');
                    $singleDay['UpSale']['today'] = $this->getSales(today(), 'upsell');

                $singleDay['GrossVolumes']['lastPayment'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();
                $singleDay['NewCustomers']['lastPayment'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['lastPayment'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();

                $singleDay['GrossVolumes']['today'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['NewCustomers']['today'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['today'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $balanceAmount = Payment::with('link')
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('currency', $currencyId)
                    ->where('status', 1)
                    ->select(DB::raw('SUM(price) as total'))
                    ->first();
                $monthlyBalance = Payment::select(DB::raw('SUM(price) as total'))
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->whereDate('created_at', '>=', Carbon::now()->firstOfMonth()->format('Y-m-d'))
                    ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
                    ->where('currency', $currencyId)
                    ->where('status', 1)->first();

                $yearlyBalance = Payment::select(DB::raw('SUM(price) as total'))
                ->whereHas('link', function ($query) use ($user) {
                    $query->where('created_by', $user->id);
                })
                ->whereDate('created_at', '>=', Carbon::now()->firstOfYear()->format('Y-m-d'))
                ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
                ->where('currency', $currencyId)
                ->where('status', 1)->first();

                $dailyUpSell = $this->getSaleByTypeDateAndCurrency('upsell', Carbon::now()->startOfDay(), Carbon::now()->endOfDay(), $currencyId, $superAdminIds=null, $user->id);
                $dailyFreshSell = $this->getSaleByTypeDateAndCurrency('fresh_sales', Carbon::now()->startOfDay(), Carbon::now()->endOfDay(), $currencyId, $superAdminIds=null, $user->id);

                $monthlyUpSell = $this->getSaleByTypeDateAndCurrency('upsell', Carbon::now()->firstOfMonth(), Carbon::now(), $currencyId, $superAdminIds=null, $user->id);
                $monthlyFreshSell = $this->getSaleByTypeDateAndCurrency('fresh_sales', Carbon::now()->firstOfMonth(), Carbon::now(), $currencyId, $superAdminIds=null, $user->id);

                $monthlyGrossBalance = Payment::select(DB::raw('SUM(price) as total'))
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->whereDate('created_at', '>=', Carbon::now()->firstOfMonth()->format('Y-m-d'))
                    ->whereDate('created_at', '<=', Carbon::now()->format('Y-m-d'))
                    ->where('currency', $currencyId)
                    ->whereIn('status', [1, 11])
                    ->first();

                $invoices = Invoice::select('id', 'invoice_no', 'custom_id', 'customer_id', 'payment_id', 'created_by', 'status', 'created_at')
                    ->with([
                        'paymentLink' => function ($query) {
                            $query->select('payment_links.id', 'payment_links.price', 'payment_links.currency')
                                ->with([
                                    'currencyCountry:id,symbol'
                                ]);
                        },
                        'customer:id,email'
                    ])
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->where('created_by', $user->id)
                    ->orderBy('id', 'DESC')
                    ->limit(3)
                    ->get();

                $refundPayments = Payment::refundCount($startDate, $endDate, $user->id, true);

                $refundPaymentsData = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'intent_id', 'created_at')
                    ->with('customer:id,email')
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->whereHas('customer', function ($query) {
                        $query->where('deleted_at', null);
                    })
                    ->whereIn('status', [9, 10, 11])
                    ->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate)
                    ->orderByDesc('id')->limit(4)->get();
            }
        } else {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $unreadPayments = Notification::select('id', 'notifiable_id', 'data', 'read_at')
            ->where('notifiable_id', Auth::id())
            ->where('data->type', 'payment')
            ->whereNull('read_at')
            ->get();

        $unreadPaymentData = $unreadPayments->map(function ($notification) {
            return $notification;
        });


        $paymentsBar['newsale'] = $freshsalesCount;
        $paymentsBar['upsell'] = $upsellCount;

        return Inertia::render('Admin/Home', [
            'page' => $page,
            'duration' => $request->dateDuration,
            'startDate' => Carbon::createFromDate($startDate)->format('d/m/Y'),
            'endDate' => Carbon::createFromDate($endDate)->format('d/m/Y'),
            'currentHour' => (int) $currentHour,
            'lastPaymentDate' => Carbon::createFromDate($selectedDate['date'])->format('d/m/Y'),
            'paymentsCount' => $paymentsBar,
            'failedPayments' => $failedPayments,
            'topCustomerBySpend' => $topCustomerBySpend,
            'grossVolume' => $grossVolume,
            'netVolumeFromSales' => $netVolumeFromSales,
            'newCustomers' => $newCustomers,
            'successfulPayments' => $successfulPayments,
            'spendPerCustomer' => $spendPerCustomer,
            'upSalesGraph' => $upSalesGraph,
            'freshSalesGraph' => $freshSalesGraph,
            'singleDay' => $singleDay,
            'monthlyGrossBalance' => (float) $monthlyGrossBalance->total,
            'balanceAmount' => (float) $balanceAmount->total,
            'monthlyBalance' => (float) $monthlyBalance->total,
            'yearlyBalance' => (float) $yearlyBalance->total,
            'dailyUpSell' => !empty($dailyUpSell->first()->sum) ? number_format($dailyUpSell->first()->sum, 2) : 0,
            'dailyFreshSell' => !empty($dailyFreshSell->first()->sum) ? number_format($dailyFreshSell->first()->sum, 2) : 0,
            'monthlyUpSell' => !empty($monthlyUpSell->first()->sum) ? number_format($monthlyUpSell->first()->sum, 2) : 0,
            'monthlyFreshSell' => !empty($monthlyFreshSell->first()->sum) ? number_format($monthlyFreshSell->first()->sum, 2) : 0,
            'customers' => $customersData,
            'invoices' => $invoices,
            'unreadPaymentData' => $unreadPaymentData,
            'refundPayments' => $refundPayments,
            'refundPaymentsData' => $refundPaymentsData,
            // 'graphSales' => $graphSales,
            'highRiskPayments' => $highRiskPayments,
            'user' => User::select('id', 'dashboard_data')->where('id', Auth::id())->first(),
        ]);
    }

    public function paymentsCountFilter($dateFrom = "", $dateTo = "")
    {
        $statuses = StripeHelper::paymentStatuses();

        $paymentSums = Payment::sumCount($statuses, $dateFrom, $dateTo);
        $paymentSums = $paymentSums->toArray();

        $payments = collect([
            'succeeded' => 0,
            'uncaptured' => 0,
            'refunded' => 0,
            'failed' => 0,
            'incomplete' => 0,
        ]);

        $payments = $payments->mapWithKeys(function ($payment, $key) use ($paymentSums) {
            if (isset($paymentSums[$key]) && !empty($paymentSums[$key])) {
                return [$key => $paymentSums[$key]];
            }
            return [$key => $payment];
        });

        return $payments;
    }

    public function userDashboardAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'graphs' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect('post/create')
                ->withErrors($validator)
                ->withInput();
        }

        // Retrieve the validated input...
        $validated = $validator->validated();
        $user = User::find(Auth::id());

        if (!empty($validated['dashboard_data'])) {
            $dashboardData = json_decode($user->dashboard_data);
            $dashboardData->graphs = $validated['graphs'];
        } else {
            $dashboardData['graphs'] = $validated['graphs'];
        }

        $user->update([
            'dashboard_data' =>  json_encode($dashboardData)
        ]);

        return Redirect::route('admin.dashboard');
    }


    public function searchAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'keyWord' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Invalid input'], 400);
        }

        $keySuggest = explode(':', $request->keyWord);
        $keyType = isset($keySuggest[0]) ? strtolower(trim($keySuggest[0])) : '';
        $keyValue = isset($keySuggest[1]) ? trim($keySuggest[1]) : $request->keyWord;

        $responses = [
            'payments' => [],
            'briefs' => [],
            'customers' => [],
            'paymentLinks' => [],
            'invoices' => [],
        ];

       $searchStrategies = [
        'last4' => function() use ($keyValue, &$responses) {
            $this->searchByLastFour($keyValue, $responses);
        },
        'email' => function() use ($keyValue, &$responses) {
            $this->searchByEmail($keyValue, $responses);
        },
        'date' => function() use ($keyValue, &$responses) {
            $this->searchByDate($keyValue, $responses);
        },
        'amount' => function() use ($keyValue, &$responses) {
            $this->searchByAmount($keyValue, $responses);
        },
        'currency' => function() use ($keyValue, &$responses) {
            $this->searchByCurrency($keyValue, $responses);
        },
        'status' => function() use ($keyValue, &$responses) {
            $this->searchByStatus($keyValue, $responses);
        },
        'is' => function() use ($keyValue, &$responses) {
            $this->searchByIs($keyValue, $responses);
        },
        'brief' => function() use ($keyValue, &$responses) {
            $this->searchByBrief($keyValue, $responses);
        },
    ];

        if (array_key_exists($keyType, $searchStrategies)) {
            $searchStrategies[$keyType]();
        } elseif (count($keySuggest) === 1 && !empty($keyType)) {
            $this->searchByGeneralTerm($keyType, $responses);
        }

        return response()->json($responses);
    }

    //Search Functions

     protected function searchByLastFour($lastFour, &$responses)
    {
        if (empty($lastFour)) return;

        $payments = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
            ->where('last_four', 'like', '%' . $lastFour . '%')
            ->with(['customer:id,email,created_at', 'link:id,price,status,created_at', 'invoice:id,payment_id,invoice_no,status,created_at'])
            ->take(3)
            ->get();

        foreach ($payments as $payment) {
            $responses['payments'][] = $payment;
            $responses['customers'][] = $payment->customer;
            $responses['paymentLinks'][] = [
                'link' => $payment->link,
                'intent_id' => $payment->intent_id
            ];
            $responses['invoices'][] = $payment->invoice;
        }
    }

    protected function searchByEmail($email, &$responses)
    {
        if (empty($email)) return;

        $customers = User::where('email', 'like', '%' . $email . '%')
            ->with(['payments' => function ($query) {
                $query->with('link')->take(3);
            }])
            ->take(3)
            ->get();

        $briefResults = collect();

        foreach ($customers as $customer) {
            $responses['customers'][] = $customer;
            foreach ($customer->payments as $payment) {
                $responses['payments'][] = $payment;
                $responses['paymentLinks'][] = [
                    'link' => $payment->link,
                    'intent_id' => $payment->intent_id
                ];
                $responses['invoices'][] = $payment->invoice;
            }

            $briefResults = $briefResults->merge($this->searchBriefsByQuery(function($query) use ($customer) {
                $query->where('customer_id', $customer->id);
            }));
        };

        $sortedResults = $briefResults->sortByDesc('created_at')->take(3);

        $responses['briefs'] = $sortedResults->values()->all();
    }

    protected function searchByDate($dateQuery, &$responses)
    {
        if (empty($dateQuery)) return;

        $keyDate = trim($dateQuery, '"');

        $operators = ['>=', '<=', '>', '<', '='];
        $operator = '=';
        foreach ($operators as $op) {
            if (strpos($keyDate, $op) === 0) {
                $operator = $op;
                $keyDate = trim(substr($keyDate, strlen($op)));
                break;
            }
        }

        $filteredDate = date('Y-m-d H:i:s', strtotime($keyDate));
        $todayDate = now()->toDateTimeString();

        $responses['payments'] = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
            ->whereBetween('created_at', [$filteredDate, $todayDate])
            ->take(3)
            ->get();

        $responses['customers'] = User::whereBetween('created_at', [$filteredDate, $todayDate])
            ->take(3)
            ->get();

        $responses['invoices'] = Invoice::whereBetween('created_at', [$filteredDate, $todayDate])
            ->take(3)
            ->get();

        $briefResults = $this->searchBriefsByQuery(function($query) use ($filteredDate, $todayDate) {
            $query->whereBetween('created_at', [$filteredDate, $todayDate]);
        });
        $sortedResults = $briefResults->sortByDesc('created_at')->take(3);
        $responses['briefs'] = $sortedResults->values()->all();

        $paymentLinks = PaymentLink::whereBetween('created_at', [$filteredDate, $todayDate])
            ->with('payments')
            ->take(3)
            ->get();

        foreach ($paymentLinks as $payLink) {
            $responses['paymentLinks'][] = [
                'link' => $payLink,
                'intent_id' => $payLink->payments->intent_id ?? ''
            ];
        }
    }

    protected function searchByAmount($amountQuery, &$responses)
    {
        if (empty($amountQuery)) return;

        $operators = ['>=', '<=', '>', '<', '='];
        $operator = '=';
        $keyAmount = $amountQuery;

        foreach ($operators as $op) {
            if (strpos($amountQuery, $op) === 0) {
                $operator = $op;
                $keyAmount = trim(substr($amountQuery, strlen($op)));
                break;
            }
        }

        $payments = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
            ->where('price', $operator, $keyAmount)
            ->take(3)
            ->get();

        $responses['payments'] = $payments;

        foreach ($payments as $payment) {
            $responses['paymentLinks'][] = [
                'link' => $payment,
                'intent_id' => $payment->intent_id ?? ''
            ];
            $responses['invoices'][] = $payment->invoice;
        }
    }

    protected function searchByCurrency($currency, &$responses)
    {
        if (empty($currency)) return;

        $payments = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
            ->whereHas('currencyCountry', function ($query) use ($currency) {
                $query->where('code', $currency)
                    ->orWhere('currency', $currency);
            })
            ->with('invoice:id,payment_id,invoice_no,status,created_at')
            ->take(3)
            ->get();

        foreach ($payments as $payment) {
            $responses['payments'][] = $payment;
            $responses['paymentLinks'][] = [
                'link' => $payment,
                'intent_id' => $payment->intent_id ?? ''
            ];
            $responses['invoices'][] = $payment->invoice;
        }
    }

    protected function searchByStatus($statusValue, &$responses)
    {
        if (empty($statusValue)) return;

        $status = StripeHelper::getStatusByValue($statusValue);

        if ($status) {
            $responses['payments'] = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
                ->where('status', $status)
                ->take(3)
                ->get();
        }
    }

    protected function searchByIs($isQuery, &$responses)
    {
        if (empty($isQuery)) return;

        $searchKey = explode(" ", $isQuery);
        $entity = strtolower($searchKey[0] ?? '');
        $value = $searchKey[1] ?? null;


        $entityStrategy = [
            'payment' => function () use ($value, &$responses){
                $this->searchIsPayment($value, $responses);
            },
            'customer_account' => function() use($value, &$responses){
                $this->searchIsCustomerAccount($value, $responses);
            },
            'customer' => function() use($value, &$responses){
                $this->searchIsCustomer($value, $responses);
            }
        ];

        if(array_key_exists($entity, $entityStrategy)){
            $entityStrategy[$entity]();
        }
    }

    protected function searchIsPayment($value, &$responses)
    {
        if ($value) {
            $payments = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
                ->where('price', $value)
                ->orWhere('intent_id', $value)
                ->orWhere('charge_id', $value)
                ->orWhere('last_four', 'like', '%' . $value . '%')
                ->orWhere('coupon_id', 'like', '%' . $value . '%')
                ->with(['currencyCountry' => function ($query) use ($value) {
                    $query->where('code', 'like', '%' . $value . '%')
                        ->orWhere('currency', 'like', '%' . $value . '%')
                        ->orWhere('country', 'like', '%' . $value . '%');
                }])
                ->take(3)
                ->get();

            $responses['payments'] = $payments;

            foreach ($payments as $payment) {
                $responses['paymentLinks'][] = [
                    'link' => $payment,
                    'intent_id' => $payment->intent_id ?? ''
                ];
                $responses['invoices'][] = $payment->invoice;
            }
        } else {
            $responses['payments'] = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
                ->latest('id')
                ->take(10)
                ->get();
        }
    }

    protected function searchIsCustomerAccount($value, &$responses)
    {
        if ($value) {
            $responses['customers'] = User::where('first_name', 'like', '%' . $value . '%')
                ->orWhere('last_name', 'like', '%' . $value . '%')
                ->orWhere('email', $value)
                ->take(3)
                ->get();
        } else {
            $responses['customers'] = User::latest('id')->take(10)->get();
        }
    }

    protected function searchIsCustomer($value, &$responses)
    {
        if ($value) {
            $responses['customers'] = User::where('email', 'like', '%' . $value . '%')
                ->orWhere('stripe_customer_id', $value)
                ->orWhere('phone', $value)
                ->orWhere('address', 'like', '%' . $value . '%')
                ->orWhere('company', 'like', '%' . $value . '%')
                ->orWhere('city', 'like', '%' . $value . '%')
                ->orWhere('zipcode', 'like', '%' . $value . '%')
                ->orWhere('country', 'like', '%' . $value . '%')
                ->orWhere('designation', 'like', '%' . $value . '%')
                ->take(3)
                ->get();
        } else {
            $responses['customers'] = User::latest('id')->take(10)->get();
        }
    }

    protected function searchByBrief($isQuery, &$responses)
    {
        if (empty($isQuery)) return;

        $searchKey = explode(" ", $isQuery);
        $entity = strtolower($searchKey[0] ?? '');
        $value = $searchKey[1] ?? null;

        $entityStrategy = [
            'logo' =>  new Brief(),
            'web' =>  new WebBrief(),
            'video' => new VideoBrief(),
            'mobile' => new MobileAppBrief(),
            'smm' => new SmmBrief(),
            'sem' => new SemBrief(),
        ];

        if (array_key_exists($entity, $entityStrategy)) {
            $briefClass = $entityStrategy[$entity];

            if ($value) {
                $results = $briefClass::where('name', 'like', '%' . $value . '%')
                    ->orWhere('email', 'like', '%' . $value . '%')
                    ->orWhere('phone', 'like', '%' . $value . '%')
                    ->orWhereHas('customer', function ($query) use ($value) {
                        $query->whereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%{$value}%"])
                            ->orWhere('email', 'like', '%' . $value . '%');
                    })
                    ->with('customer')
                    ->latest('created_at')
                    ->limit(5)
                    ->get()
                    ->map(function ($brief)use($entity) {
                        $brief->brief_type = $entity;
                        return $brief;
                    });

                $responses['briefs'] = $results;
            }
        }
    }


    protected function fetchBriefResults($query, $table)
    {
        return $query
            ->with('customer')
            ->latest('created_at')
            ->take(3)
            ->get()
            ->map(function ($brief) use ($table) {
                $brief->brief_type = $table['brief_type'];
                return $brief;
            });
    }

    protected function searchBriefsByQuery(callable $queryLogic) {
        $results = collect();

        $tables = [
            ['model' => Brief::class, 'brief_type' => 'logo'],
            ['model' => WebBrief::class, 'brief_type' => 'web'],
            ['model' => VideoBrief::class, 'brief_type' => 'video'],
            ['model' => MobileAppBrief::class, 'brief_type' => 'mobile'],
            ['model' => SmmBrief::class, 'brief_type' => 'smm'],
            ['model' => SemBrief::class, 'brief_type' => 'sem'],
        ];

        foreach ($tables as $table) {
            $query = $table['model']::query();

            $queryLogic($query);

            $tableResults = $this->fetchBriefResults($query, $table);
            $results = $results->concat($tableResults);
        }

        return $results;
    }

    protected function searchByGeneralTerm($term, &$responses)
    {
        if (empty($term)) return;

        $payments = Payment::select('id', 'customer_id', 'payment_link_id', 'price', 'intent_id', 'status', 'charge_id', 'last_four', 'created_at')
            ->where('price', 'like', '%' . $term . '%')
            ->orWhere('intent_id', 'like', '%' . $term . '%')
            ->orWhere('charge_id', 'like', '%' . $term . '%')
            ->orWhere('last_four', 'like', '%' . $term . '%')
            ->orWhere('coupon_id', 'like', '%' . $term . '%')
            ->orWhereHas('currencyCountry', function ($query) use ($term) {
                $query->where('code', 'like', '%' . $term . '%')
                    ->orWhere('currency', 'like', '%' . $term . '%')
                    ->orWhere('country', 'like', '%' . $term . '%');
            })
            ->with(['customer:id,email,created_at', 'link:id,price,status,created_at', 'invoice:id,invoice_no,status,created_at'])
            ->take(3)
            ->get();

        $responses['payments'] = $payments;

        $customers = User::where('first_name', 'like', '%' . $term . '%')
            ->orWhere('last_name', 'like', '%' . $term . '%')
            ->orWhere('email', 'like', '%' . $term . '%')
            ->orWhere('stripe_customer_id', 'like', '%' . $term . '%')
            ->orWhere('phone', 'like', '%' . $term . '%')
            ->orWhere('address', 'like', '%' . $term . '%')
            ->orWhere('company', 'like', '%' . $term . '%')
            ->orWhere('city', 'like', '%' . $term . '%')
            ->orWhere('zipcode', 'like', '%' . $term . '%')
            ->orWhere('country', 'like', '%' . $term . '%')
            ->orWhere('designation', 'like', '%' . $term . '%')
            ->with(['payments' => function ($query) {
                $query->with('link:id,price,status,created_at');
            }])
            ->take(3)
            ->get();

        $responses['customers'] = $customers;

        $responses['invoices'] = Invoice::where('invoice_no', 'like', '%' . $term . '%')
            ->orWhere('custom_id', 'like', '%' . $term . '%')
            ->select('id', 'invoice_no', 'status', 'created_at')
            ->take(3)
            ->get();

        foreach ($payments as $payment) {
            $responses['paymentLinks'][] = [
                'link' => $payment->link,
                'intent_id' => $payment->intent_id ?? ''
            ];
        }

        foreach ($customers as $customer) {
            foreach ($customer->payments as $payment) {
                if (count($responses['payments']) < 3) {
                    $responses['payments'][] = $payment;
                }
                if (count($responses['paymentLinks']) < 3) {
                    $responses['paymentLinks'][] = [
                        'link' => $payment->link,
                        'intent_id' => $payment->intent_id
                    ];
                }
                if ($payment->invoice && count($responses['invoices']) < 3) {
                    $responses['invoices'][] = $payment->invoice;
                }
            }
        }


        $briefResults = $this->searchBriefsByQuery(function($query) use ($term) {
            $query->where('name', 'like', '%' . $term . '%')
                    ->orWhere('email', 'like', '%' . $term . '%')
                    ->orWhere('phone', 'like', '%' . $term . '%')
                    ->orWhereHas('customer', function ($query) use ($term) {
                        $query->whereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%{$term}%"])
                            ->orWhere('email', 'like', '%' . $term . '%');
                    });
        });
        $sortedResults = $briefResults->sortByDesc('created_at')->take(3);
        $responses['briefs'] = $sortedResults->values()->all();
    }

    public function paymentsCountFilterById($dateFrom = "", $dateTo = "", $forCurrentUser = true)
    {
        $user = Auth::user();
        $statuses = StripeHelper::paymentStatuses();

        if ($forCurrentUser) {
            $paymentSums = Payment::sumCountbyId($statuses, $dateFrom, $dateTo, $user->id);
        } else {
            $paymentSums = Payment::sumCountbyId($statuses, $dateFrom, $dateTo);
        }

        $paymentSums = $paymentSums->toArray();
        // dd($paymentSums);
        $payments = collect([
            'succeeded' => 0,
            'uncaptured' => 0,
            'refunded' => 0,
            'failed' => 0,
            'incomplete' => 0,
        ]);
        $payments = $payments->mapWithKeys(function ($payment, $key) use ($paymentSums) {
            if (isset($paymentSums[$key]) && !empty($paymentSums[$key])) {
                return [$key => $paymentSums[$key]];
            }
            return [$key => $payment];
        });

        return $payments;
    }

    public function payouts()
    {
        $currencies = Payment::select('id', DB::raw('SUM(price) as total'), 'currency')
            ->with(['currencyCountry' => function ($currency) {
                $currency->select('id', 'code', 'symbol');
            }])
            ->where('status', 1)
            ->groupBy('currency')
            ->orderByRaw("CASE WHEN currency = 170 THEN 0 ELSE 1 END")
            ->orderBy('currency', 'ASC')
            ->get();

        return Inertia::render('Admin/Payouts/Detail', [
            'currencies' => $currencies,
        ]);
    }


    /**
     *
     * Function's Optimization is incomplete
     *
     */

    public function dailyGraph(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $roles = RolesModel::exceptedRoles(['Super Admin']);

        $user = Auth::user();

        if ($user->hasRole('Super Admin')) {
            if (!empty($request->selectedDate)) {
                $selectedDate['date'] = $request->selectedDate;
                // $selectedDate['total'] =  Payment::select(DB::raw('COUNT(status) as total'), DB::raw('DATE(created_at) as date'))->where('status', 1)->whereDate('created_at', '=', $request->selectedDate)->groupBy('date')->first();
                $selectedDate['total'] = $this->getLastPaymentDate(1, $request->selectedDate);
            } else {
                // $lastPaymentDate = Payment::select(DB::raw('COUNT(status) as total'), DB::raw('DATE(created_at) as date'))->where('status', 1)->whereDate('created_at', '!=', Carbon::now()->format('Y-m-d'))->groupBy('date')->latest('created_at')->first();

                // $lastPaymentDate = Payment::selectRaw('COUNT(status) as total, DATE(created_at) as date')
                //                                     ->where('status', 1)
                //                                     ->whereDate('created_at', '<>', today())
                //                                     ->groupBy('date')
                //                                     ->latest('created_at')
                //                                     ->first();


                $lastPaymentDate = $this->getLastPaymentDate(1);


                $selectedDate['date'] = $lastPaymentDate->date ?? 0;
                $selectedDate['total'] = $lastPaymentDate->total ?? 0;
            }


            $singleDay['NewSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'fresh_sales');
            $singleDay['NewSale']['today'] = $this->getSales(today(), 'fresh_sales');

            $singleDay['UpSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'upsell');
            $singleDay['UpSale']['today'] = $this->getSales(today(), 'upsell');


            // $singleDay['NewSale']['lastPayment'] = PaymentLink::selectRaw('SUM(price) as total, HOUR(created_at) as hour')
            //                             ->where('status', 1)
            //                             ->whereDate('created_at', '=', $selectedDate['date'])
            //                             ->groupBy('hour')
            //                             ->get();


            $singleDay['GrossVolumes']['lastPayment'] = Payment::select(DB::raw('SUM(price) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', $selectedDate['date'])->groupBy('hour')->get();
            $singleDay['NewCustomers']['lastPayment'] = User::select(DB::raw('COUNT(id) as total'), DB::raw('HOUR(created_at) as hour'))->role('customer')->whereDate('created_at', '=', $selectedDate['date'])->groupBy('hour')->get();
            $singleDay['SuccessfulPayments']['lastPayment'] = Payment::select(DB::raw('COUNT(status) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', $selectedDate['date'])->groupBy('hour')->get();
            $singleDay['GrossVolumes']['today'] = Payment::select(DB::raw('SUM(price) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))->groupBy('hour')->get();
            $singleDay['NewCustomers']['today'] = User::select(DB::raw('COUNT(id) as total'), DB::raw('HOUR(created_at) as hour'))->role('customer')->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))->groupBy('hour')->get();
            $singleDay['SuccessfulPayments']['today'] = Payment::select(DB::raw('COUNT(status) as total'), DB::raw('HOUR(created_at) as hour'))->where('status', 1)->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))->groupBy('hour')->get();
        } else if ($user->hasAnyRole($roles)) {
            if ($user->can('Payment-Global')) {

                $superAdminIds = User::whereHas('roles', function ($query) {
                    $query->where('name', 'Super Admin');
                })->pluck('id');

                // $lastPaymentDate = $this->getLastPaymentDate(1, $request->selectedDate, 'Super Admin');

                // dd($lastPaymentDate);

                if (!empty($request->selectedDate)) {
                    $selectedDate['date'] = $request->selectedDate;
                    $selectedDate['total'] = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date')
                    )
                        ->whereHas('link', function ($query) use ($superAdminIds) {
                            $query->whereNotIn('created_by', $superAdminIds);
                        })
                        ->where('status', 1)
                        // ->where('customer_id', '=', $user->id)
                        ->whereDate('created_at', '=', $request->selectedDate)
                        ->groupBy('date')
                        ->first();
                } else {
                    $selectedDate['date'] = $lastPaymentDate->date ?? 0;
                    $selectedDate['total'] = $lastPaymentDate->total ?? 0;
                }


                $singleDay['NewSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'fresh_sales');
                $singleDay['NewSale']['today'] = $this->getSales(today(), 'fresh_sales');

                $singleDay['UpSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'upsell');
                $singleDay['UpSale']['today'] = $this->getSales(today(), 'upsell');


                $singleDay['GrossVolumes']['lastPayment'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();
                $singleDay['NewCustomers']['lastPayment'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['lastPayment'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();

                $singleDay['GrossVolumes']['today'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['NewCustomers']['today'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['today'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($superAdminIds) {
                        $query->whereNotIn('created_by', $superAdminIds);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
            } else {

                if (!empty($request->selectedDate)) {
                    $selectedDate['date'] = $request->selectedDate;
                    $selectedDate['total'] = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date')
                    )
                        ->whereHas('link', function ($query) use ($user) {
                            $query->where('created_by', $user->id);
                        })
                        ->where('status', 1)
                        // ->where('customer_id', '=', $user->id)
                        ->whereDate('created_at', '=', $request->selectedDate)
                        ->groupBy('date')
                        ->first();
                } else {
                    $lastPaymentDate = Payment::select(
                        DB::raw('COUNT(status) as total'),
                        DB::raw('DATE(created_at) as date'),
                    )
                        ->whereHas('link', function ($query) use ($user) {
                            $query->where('created_by', $user->id);
                        })
                        ->where('status', 1)
                        ->whereDate('created_at', '!=', Carbon::now()->format('Y-m-d'))
                        ->orderBy('created_at', 'desc')
                        ->groupBy('date')
                        ->first();

                    $selectedDate['date'] = $lastPaymentDate->date ?? 0;
                    $selectedDate['total'] = $lastPaymentDate->total ?? 0;
                }

                $singleDay['NewSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'fresh_sales');
                $singleDay['NewSale']['today'] = $this->getSales(today(), 'fresh_sales');

                $singleDay['UpSale']['lastPayment'] = $this->getSales($selectedDate['date'], 'upsell');
                $singleDay['UpSale']['today'] = $this->getSales(today(), 'upsell');


                $singleDay['GrossVolumes']['lastPayment'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();
                $singleDay['NewCustomers']['lastPayment'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['lastPayment'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', $selectedDate['date'])
                    ->groupBy('hour', 'created_at')
                    ->get();

                $singleDay['GrossVolumes']['today'] = Payment::select(
                    DB::raw('SUM(price) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['NewCustomers']['today'] = User::select(
                    DB::raw('COUNT(id) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->role('customer')
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
                $singleDay['SuccessfulPayments']['today'] = Payment::select(
                    DB::raw('COUNT(status) as total'),
                    DB::raw('HOUR(created_at) as hour')
                )
                    ->whereHas('link', function ($query) use ($user) {
                        $query->where('created_by', $user->id);
                    })
                    ->where('status', 1)
                    ->whereDate('created_at', '=', Carbon::now()->format('Y-m-d'))
                    ->groupBy('hour')
                    ->get();
            }
        }

        return response()->json([
            'status' => true,
            'data' => $singleDay,
        ]);
    }


    private function getLastPaymentDate($status, $date = null, $roleExcluded = null)
    {
        $query = Payment::selectRaw('COUNT(status) as total, DATE(created_at) as date')
            ->where('status', $status);

        if ($date) {
            $query->whereDate('created_at', '=', $date);
        } else {
            $query->whereDate('created_at', '!=', today())->latest('created_at');
        }

        if($roleExcluded) {
            $roleIDs = RolesModel::getUserIDWithRole($roleExcluded);
        }

        $query->groupBy('date');

        return $query->first();
    }

    private function getSales($date, $salesType)
    {
        return PaymentLink::selectRaw('SUM(price) as total, HOUR(created_at) as hour')
            ->where('status', 1)
            ->where('sale_type', $salesType)
            ->whereDate('created_at', $date)
            ->groupBy('hour')
            ->get();
    }

    // private function getGraphSales()
    // {
    //     return PaymentLink::selectRaw('SUM(price) as count, sale_type as name')
    //         ->where('status', 1)
    //         ->groupBy('sale_type')
    //         ->get();
    // }

    private static function getSaleByTypeAndDate($saleType, $startDate, $endDate)
    {
        $baseQuery = Payment::where('status', 1);

        if($saleType) {
            $baseQuery->whereHas('paymentLink', function($query) use ($saleType) {
                $query->where('sale_type', $saleType);
            });
        }

        if($startDate && $endDate) {
            $baseQuery = $baseQuery->whereBetween('created_at', [$startDate, $endDate]);
        }

        $baseQuery = $baseQuery->selectRaw('COUNT(price) as total, SUM(price) as sum, DATE(created_at) as date')
        ->groupBy('date')
        ->get();

        return $baseQuery;
    }

    private static function getSaleByTypeDateAndCurrency($saleType, $startDate, $endDate, $currencyId, $superAdminIds="", $userId="")
    {
        $baseQuery = Payment::where('status', 1);

        if($saleType) {
            $baseQuery->whereHas('paymentLink', function($query) use ($saleType, $superAdminIds, $userId) {

                $query->where('sale_type', $saleType);

                if (!empty($superAdminIds)) {
                    $query->whereNotIn('created_by', $superAdminIds);
                }

                if (!empty($userId)) {
                    $query->where('created_by', $userId);
                }
            });
        }

        if($startDate && $endDate) {
            $baseQuery = $baseQuery->whereBetween('created_at', [$startDate, $endDate]);
        }

        if($currencyId) {
            $baseQuery = $baseQuery->where('currency', $currencyId);
        }

        $baseQuery = $baseQuery->selectRaw('COUNT(price) as total, SUM(price) as sum, DATE(created_at) as date')
        ->get();

        return $baseQuery;
    }

    private function getSaleTypeCountFilter($saleType, $startDate, $endDate)
    {
        return Payment::query()
        ->selectRaw('SUM(payments.price) as count, link.sale_type as name')
        ->join('payment_links as link', 'payments.payment_link_id', '=', 'link.id')
        ->where('link.sale_type', $saleType)
        ->where('payments.status', 1)
        ->whereDate('payments.created_at', '>=', $startDate)
        ->whereDate('payments.created_at', '<=', $endDate)
        ->groupBy('link.sale_type')
        ->first();
    }

    // private function getSaleTypeCountFilter($saleType, $startDate, $endDate)
    // {
    //     return Payment::join('payment_links', 'payments.payment_link_id', '=', 'payment_links.id')
    //     ->whereDate('payments.created_at', '>=', $startDate)
    //     ->whereDate('payments.created_at', '<=', $endDate)
    //     ->where('payment_links.sale_type', $saleType)
    //     ->where('payments.status', 1)
    //     ->selectRaw('COUNT(payments.id) as total, DATE(payments.created_at) as date')
    //     ->groupBy('date')
    //     ->get();
    // }

    private function getSaleTypeCountFilterById($saleType, $startDate, $endDate)
    {
        $user = Auth::user();

        return Payment::query()
            ->selectRaw('SUM(payments.price) as count, link.sale_type as name')
            ->join('payment_links as link', 'payments.payment_link_id', '=', 'link.id')
            ->where('link.sale_type', $saleType)
            ->where('payments.status', 1)
            ->whereDate('payments.created_at', '>=', $startDate)
            ->whereDate('payments.created_at', '<=', $endDate)
            ->where('link.created_by', $user->id)
            ->groupBy('link.sale_type')
            ->first();
    }



    // private function getSalesData($salesType, $startDate, $endDate = null, $groupBy = 'hour', $aggregateFunction = 'SUM')
    // {
    //     // Initialize the query
    //     $query = PaymentLink::selectRaw(
    //         "$aggregateFunction(price) as total, " .
    //         ($groupBy === 'hour' ? 'HOUR(created_at) as hour' : 'DATE(created_at) as date')
    //     )
    //     ->where('status', 1)
    //     ->where('sale_type', $salesType);

    //     // Add date conditions
    //     if ($endDate) {
    //         $query->whereDate('created_at', '>=', $startDate)
    //               ->whereDate('created_at', '<=', $endDate);
    //     } else {
    //         $query->whereDate('created_at', $startDate);
    //     }

    //     // Add group by clause
    //     if ($groupBy === 'hour') {
    //         $query->groupBy('hour');
    //     } else {
    //         $query->groupBy('date');
    //     }

    //     // Execute the query and return the results
    //     return $query->get();
    // }


    private function getSelectedDate($request) {
        if (!empty($request->selectedDate)) {
            $selectedDate['date'] = $request->selectedDate;
            $selectedDate['total'] = $this->getPaymentTotalByDate($request->selectedDate, 1);
        } else {
            $lastPaymentDate = $this->getLastPaymentDate(1);
            $selectedDate['date'] = $lastPaymentDate->date ?? 0;
            $selectedDate['total'] = $lastPaymentDate->total ?? 0;
        }
        return $selectedDate;
    }

    private function getHighRiskPayments($startDate, $endDate)
    {
        $highRiskPayments = [];

        $highRiskPaymentsData = Payment::whereBetween('created_at', [$startDate, $endDate])->get();

        foreach ($highRiskPaymentsData as $highRiskPayment) {
            $entity = json_decode($highRiskPayment->charge);

            if (isset($entity->outcome) && isset($entity->outcome->risk_score)) {
                $highRiskPayments[] = [
                    'total' => $entity->outcome->risk_score,
                    'date' => $highRiskPayment->created_at->format('Y-m-d'),
                ];
            }
        }

        return $highRiskPayments;
    }
}
