<?php

namespace App\Http\Controllers\Admin;

use App\Helper\Helper\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\CouponRequest;
use App\Models\CountryCurrencies;
use App\Models\CouponLog;
use App\Services\CouponService;
use App\Services\MetadataService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class CouponController extends Controller
{

    private $couponService;

    private $columns = [
        ["key" => "name", "component" => 'NameColumn', "text" => "Coupon", "visible" => true, "fixed" => true],
        ["key" => "discount", "component" => 'TermsColumn', "text" => "Terms", "visible" => true, "fixed" => true],
        ["key" => "duration", "component" => 'DurationColumn', "text" => "Duration", "visible" => true, "fixed" => true],
        ["key" => "expires", "component" => 'ExpiresColumn', "text" => "Expires", "visible" => true, "fixed" => true],
    ];

    public function __construct(CouponService $couponService)
    {
        $this->couponService = $couponService;
    }

    public function index(Request $request)
    {
        $page["title"] = "Coupons List - ProductModule";
        $columns = $this->columns;
        $statuses = ['all', 'available', 'archived'];
        $coupons = $this->couponService->getCoupons($request);

        return Inertia::render('Admin/Product/CouponList', [
            'columns' => $columns,
            'page' => $page,
            'statuses' => $statuses,
            'coupons' => $coupons,
        ]);
    }

    public function detail(Request $request)
    {
        $request->validate(['id' => 'required|integer',]);
        $page["title"] = "Coupon Detail - ProductModule";
        $coupon = $this->couponService->getCouponDetail($request->id);

        if (!$coupon) {
            return $this->jsonError('Coupon not found', 404);
        }

        return Inertia::render('Admin/Product/CouponDetail', [
            'coupon' => $coupon,
            'page' => $page,
        ]);
    }

    public function create()
    {
        $countries = CountryCurrencies::select('id', 'code', 'symbol')->get();

        return Inertia::render('Admin/Product/CouponCreate',
            [
                'method' => 'add',
                'countries' => $countries,
            ]
        );
    }

    public function store(CouponRequest $request)
    {
        $coupon = $this->couponService->create($request);

        if ($coupon) {
            $this->logCouponActivity($coupon, $request, 'created');
            $this->sendNotification($coupon, 'Coupon Added ' . $request->description, 'Coupon Add Successful');

            Session::flash('success', true);
            Session::flash('message', 'New Coupon Created.');

              return $this->jsonSuccess('Coupon created successfully.');
        }

        return $this->jsonError('Error creating coupon.');
    }

    public function edit(Request $request, $id)
    {
        $coupon =  $this->couponService->getCouponById($id);

        if ($coupon->date_from && $coupon->date_to) {
            $coupon->duration_months = $coupon->date_to->diffInMonths($coupon->date_from);
        }

        $countries = CountryCurrencies::select('id', 'code', 'symbol')->get();

        return Inertia::render('Admin/Product/CouponCreate', [
            'method' => 'edit',
            'coupon' => $coupon,
            'countries' => $countries,
        ]);
    }

    public function update(CouponRequest $request)
    {
        $coupon = $this->couponService->update($request);

        if ($coupon) {
            $this->logCouponActivity($coupon, $request, 'updated');
            $this->sendNotification($coupon, 'Coupon Updated ' . $request->description, 'Coupon Updated Successful');

            Session::flash('success', true);
            Session::flash('message', 'Coupon Updated.');

            return $this->jsonSuccess('Coupon updated successfully.');
        }

        return $this->jsonError('Error updating coupon.');
    }

    public function delete(Request $request)
    {
        $coupon = $this->couponService->deleteById($request->id);

        if (!$coupon) {
            return $this->jsonError('Coupon not found', 404);
        }

        $this->sendNotification($request, 'Coupon Deleted ' . $request->description, 'Coupon Deleted Successfully');

        Session::flash('success', true);
        Session::flash('message', 'Coupon Deleted.');

        return $this->jsonSuccess('Coupon deleted successfully.');
    }

    public function active(Request $request)
    {
        $coupon = $this->couponService->getCouponById($request->id);

        if (!$coupon) {
            return $this->jsonError('Coupon not found', 404);
        }

        if ($coupon->duration == "multiple_months" && $coupon->date_from && $coupon->date_to) {

            $monthsDifference = Carbon::parse($coupon->date_from)->diffInMonths(Carbon::parse($coupon->date_to));

            $coupon->date_from = Carbon::today();
            $coupon->date_to = Carbon::today()->addMonths($monthsDifference);
        } else {
            $coupon->date_from = null;
            $coupon->date_to = null;
        }

        $coupon->save();

        Session::flash('success', true);
        Session::flash('message', 'Coupon Activated.');

        return $this->jsonSuccess('Coupon Activated successfully.');
    }

    public function updateMetadata(Request $request, $id)
    {
        $coupon = $this->couponService->getCouponById($id);

        if ($request->filled('metadata')) {
            $metadataService = new MetadataService($coupon, 'data');
            $metadataService->update($request->metadata);
        }
    }

    public function redeemCoupon(Request $request)
    {
        $validator = Validator::make($request->all(), [
                'couponId' => 'required|exists:coupons,name'
            ], [
                'couponId.required' => 'Please enter your coupon to redeem!',
                'couponId.exists' => 'Coupon is invalid!'
            ]);

        if ($validator->fails()) {
            $data['errors'] = $validator->errors();
            return response()->json($data, 422);
        }

        $coupon = $this->couponService->getCouponByName($request->couponId);

        if (!$coupon) {
            return $this->jsonError('Coupon not found', 404);
        }

        $couponData = $this->getCouponData($coupon);

        return $this->jsonSuccess('Coupon retrieved successfully.', ['coupon' => $couponData]);
    }

    private function logCouponActivity($coupon, $request, $type)
    {
        CouponLog::create([
            'coupon_id' => $coupon->id,
            'activity' => "Coupon $type.",
            'type' => $type,
            'request' => response()->json($request->all())->content(),
            'response' => response()->json($coupon)->content(),
            'created_by' => $request->user()->id,
        ]);
    }

    private function sendNotification($coupon, $description, $message)
    {
        NotificationHelper::notify('coupon', $description, [
            'id' => $coupon->id,
            'message' => $message,
        ], Auth::user()->id);
    }

    private function getCouponData($coupon)
    {
        return [
            'id' => $coupon->id,
            'name' => $coupon->name,
            'description' => $coupon->description,
            'discount' => $coupon->discount,
            'discountType' => $coupon->discount_type,
            'currency' => [
                'id' => $coupon->currencyCountry->id,
                'symbol' => $coupon->currencyCountry->symbol,
                'code' => $coupon->currencyCountry->code,
            ],
        ];
    }

    private function jsonSuccess($message, $data = [])
    {
        return response()->json(array_merge(['success' => true, 'message' => $message], $data));
    }

    private function jsonError($message, $statusCode = 500)
    {
        return response()->json(['success' => false, 'message' => $message], $statusCode);
    }
}
