<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Helper\BriefHelper;
use App\Helper\FileHelper;
use App\Helper\LogsHelper;
use App\Helper\PDFHelper;
use App\Http\Controllers\Controller;
use App\Models\BriefActivity;
use App\Models\BriefLink;
use App\Models\User;
use App\Models\VideoBrief;
use App\Models\VisitedBrief;
use App\Services\ApiService;
use App\Services\BinApiService;
use App\Services\BriefActivityService;
use App\Services\BriefLogService;
use App\Services\IpGeoLocationService;
use App\Services\LogService;
use App\Services\VideoBriefService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Illuminate\Support\Facades\Validator;

class VideoBriefController extends Controller
{

    private $videoBrief, $apiService, $briefLogService, $videoBriefService, $briefLink, $ipLocationService, $neutrinoApi, $briefActivityService, $logRepository;
    private $logsDirectory = 'videoBrief', $linkLongDirectory = 'brief_link';

    public function __construct(VideoBrief $videoBrief, ApiService $apiService, BriefLogService $briefLogService, VideoBriefService $videoBriefService, BriefLink $briefLink, IpGeoLocationService $ipLocationService,  BinApiService $neutrinoApi, BriefActivityService $briefActivityService, LogService $logRepository)
    {
        $this->videoBrief = $videoBrief;
        $this->apiService = $apiService;
        $this->briefLogService = $briefLogService;
        $this->videoBriefService = $videoBriefService;
        $this->briefLink = $briefLink;
        $this->ipLocationService = $ipLocationService;
        $this->neutrinoApi = $neutrinoApi;
        $this->briefActivityService = $briefActivityService;
        $this->logRepository = $logRepository;
    }

    public function list(Request $request)
    {
        $briefs = $this->videoBriefService->list($request);

        $columns = [
            ["key" => "token", "component" => "TokenColumn", "text" => "Brief Form Link", "visible" => true, "fixed" => true],
            ["key" => "name", "component" => "NameColumn", "text" => "Customer Name", "visible" => true],
            ["key" => "phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ["key" => "brief_link.customer.company", "component" => "CompanyColumn", "text" => "Company Name", "visible" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ["key" => "industry", "component" => "IndustryColumn", "text" => "Industry", "visible" => false],
            ['key' => "brief_link.salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        $page = [
            'title' => 'Video Briefs List',
            'heading' => 'Briefs',
            'detailRoute' => 'video.brief-detail.detail',
            'tabs' => [
                [
                    'title' => 'All Briefs',
                    'route' => 'video.brief.list',
                    'isActive' => true,
                ],
            ],
        ];

        return Inertia::render('Admin/Brief/Brief/VideoList', [
            'briefs' => $briefs,
            'page' => $page,
            'columns' => $columns,
        ]);
    }


    public function store(Request $request)
    {

        $rules = [
            'email' => 'required|email|max:255',
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:255',
            'project_name' => 'required|string|max:400',
            'industry' => 'required|string|max:255',
            'website' => 'required|string|max:400',
            'social_media' => 'required|string|max:400',
            'business_description' => 'required|string|max:400',
            'business_services' => 'required|string|max:400',
            'audience' => 'required|string|max:400',
            'competitor' => 'required|string|max:400',
            'u_s_p' => 'required|string|max:400',
            'comments' => 'required|string|max:65535',
            'video_purpose' => 'required|string|max:400',
            'video_duration' => 'required|string|max:400',
            'reference' => 'required|string|max:400',
            'style' => 'required|string|max:400',
            'color' => 'required|string|max:400',
            'voice' => 'required|array',
            'artist_accent' => 'required|string|max:400',
            'animation_style' => 'required|string|max:400',
            'reference_video' => 'required|string|max:400',
            'deadline' => 'required|date_format:Y-m-d',
            'additional_comments' => 'required|string|max:65535 ',
        ];

        $messages = [
            'email.required' => 'Email is required.',
            'email.email' => 'Email must be a valid email address.',
            'name.required' => 'Name is required.',
            'phone.required' => 'Phone number is required.',
            'project_name.required' => 'Project name is required.',
            'industry.required' => 'Industry is required.',
            'website.required' => 'Website is required.',
            'social_media.required' => 'Social media is required.',
            'business_description.required' => 'Business description is required.',
            'business_services.required' => 'Business services is required.',
            'audience.required' => 'Target audience is required.',
            'competitor.required' => 'Competitor is required.',
            'u_s_p.required' => 'Unique selling proposition is required.',
            'comments.required' => 'Comments is required.',
            'video_purpose.required' => 'Video purpose is required.',
            'video_duration.required' => 'Video duration is required.',
            'reference.required' => 'Reference is required.',
            'style.required' => 'Style is required.',
            'color.required' => 'Color is required.',
            'voice.required' => 'Voiceover is required.',
            'artist_accent.required' => 'Artist accent is required.',
            'animation_style.required' => 'Animation style is required.',
            'reference_video.required' => 'Reference video is required.',
            'deadline.required' => 'Completion date is required.',
            'deadline.date_format' => 'Completion date must be in the format YYYY-MM-DD.',
            'additional_comments.required' => 'Additional comments is required.',
        ];

        try {
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $nameParts = explode(' ', $request->name, 2);
            $firstName = $nameParts[0];
            $lastName = isset($nameParts[1]) ? $nameParts[1] : '';

            if($request->customer_id){
                $existingUser = User::find($request->customer_id);
                if ($existingUser) {
                    $request->merge(['customer_id' => $existingUser->id]);
                    $newEmail = $request->email;

                    if ($existingUser->email !== $newEmail) {
                        $extraEmails = $existingUser->extra_emails ? json_decode($existingUser->extra_emails, true) : [];

                        if (!in_array($newEmail, $extraEmails)) {
                            $extraEmails[] = $newEmail;
                            $existingUser->extra_emails = json_encode($extraEmails);
                            $existingUser->save();
                        }
                    }
                }

            } else {
                $existingUser = User::where('email', $request->email)->first();
                if($existingUser){
                    $request->merge(['customer_id' => $existingUser->id]);
                } else {
                    $user = User::whereHas('roles', function ($query) {
                        $query->where('name', 'Super Admin');
                    })->first();

                    $customer = User::create([
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'email' => $request->email,
                        'phone' => $request->phone,
                        'password' => Hash::make('12345678'),
                        'created_by' => $user->id
                    ]);

                    $customer->assignRole('Customer');

                    $log = array(
                        'activity' => 'Customer was Created',
                        'type' => 'customer.created',
                        'request' => response()->json($customer)->content(),
                        'response' => 201,
                        'code' => response()->json($customer)->status(),
                        'loggable_id' => $customer->id,
                        'created_by' => $customer->id,
                    );
                    $this->logRepository->log('customer', $log);

                    $request->merge(['customer_id' => $customer->id]);
                }

            }

            $brief = $this->videoBriefService->create($request->all());
            $this->apiService->fetchActivityAndCreateFile($request, $brief['encoded_id']);

            $logData = [
                'activity' => 'Video Brief has been created by ' . $brief['email'] . '.',
                'request' => $request,
                'response' => response()->json($brief)->content(),
                'created_by' => $request->customer_id,
                'code' => 200,
            ];

            LogsHelper::log(array_merge($logData, [
                'class_id' => $brief['brief_link_id'],
                'class_type' => get_class(new BriefLink),
                'type' => 'link.submitted',
            ]), $this->linkLongDirectory);

            LogsHelper::log(array_merge($logData, [
                'class_id' => $brief['id'],
                'class_type' => get_class(new VideoBrief),
                'type' => 'created',
            ]), $this->logsDirectory);

            $customerLog = array(
                'activity' => 'Video Brief has been Created',
                'type' => 'customer.brief',
                'request' => response()->json($brief)->content(),
                'response' => 201,
                'code' => response()->json($brief)->status(),
                'loggable_id' => $request->customer_id,
                'created_by' => $request->customer_id,
            );
            $this->logRepository->log('customer', $customerLog);

            $this->briefActivityService->create($request, $brief['id'], 'video_brief', 'create', 'Brief Created Successfully', $request->fingerprint);

            if($brief['encoded_id']){
                $foundBrief = $this->briefLink->where('id', $brief['brief_link_id'])->first();
                $foundBrief->customer_id = $request->customer_id;
                $foundBrief->status = "submitted";
                $foundBrief->save();
            }

            $locationResponse = $this->ipLocationService->ipLocation($request->ip());
            $locationData = json_decode($locationResponse, true);
            $locationData['geocode-reverse'] = '';

            $latitude = $locationData['latitude'] ?? null;
            $longitude = $locationData['longitude'] ?? null;

            if ($latitude && $longitude) {
                    $reverseGeocodeData = $this->neutrinoApi->reverseGeocode($latitude, $longitude);
                    $locationData['geocode-reverse'] = $reverseGeocodeData ?? '';
            }

            if($brief['id']){
                $this->videoBrief->find($brief['id'])
                ->update([
                    'location' => $locationData,
                    'device' => $this->neutrinoApi->userAgent($request->header('User-Agent')),
                ]);
            }

            return redirect()->route('video.brief.success')->with('success', 'Brief created successfully');
        } catch (Exception $e) {

            LogsHelper::log([
                'activity' => 'Error during brief creation.',
                'request' => $request,
                'response' => $e->getMessage(),
                'created_by' => $request->customer_id ?? null,
                'code' => 500,
                'type' => 'link.error',
                'class_id' => $request->brief_link_id,
                'class_type' => get_class(new BriefLink),
            ], $this->linkLongDirectory);

            return redirect()->back()->with('error', $e->getMessage());
        }

    }

    public function success()
    {
        $pageProperties = [
            'heading' => 'Video Design Questionnaire',
            'description' => ['Your response has been recorded. Thank you for providing the necessary details.'],
        ];

        return Inertia::render('BriefSuccess', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function delete(Request $request)
    {
        $rules = [
            'id' => 'nullable|exists:video_briefs,id',
            'ids' => 'nullable|array',
            'ids.*' => 'exists:video_briefs,id',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            if ($request->filled('id')) {
                $videobrief = VideoBrief::findOrFail($request->id);
                $videobrief->delete();
            } else if ($request->filled('ids')) {
                VideoBrief::destroy($request->ids);
            }

            if (is_array($request->ids)) {
                foreach ($request->ids as $id) {
                    $this->briefActivityService->create($request, $id, 'video_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
                }
            } else {
                $this->briefActivityService->create($request, $request->id, 'video_brief', 'delete', 'Brief Deleted Successfully', $request->fingerprint);
            }

            return response()->json([
                'success' => true,
                'message' => "Brief(s) successfully deleted.",
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => "An error occurred while deleting brief(s).",
            ], 500);
        }

    }

    public function downloadBrief(Request $request, $encoded_id, $fingerprint)
    {
        try {
            $formData = $this->videoBriefService->fetchByEncodedID($encoded_id, true);

            $formData['voice'] = $this->stringToArray($formData['voice']);

            $htmlContent = Blade::render('brief.video_download', [
                'formData' => $formData,
            ]);

            $page = 'A1';
            $pdfOutput = PDFHelper::generatePDF($htmlContent, $page);
            $imageData = $pdfOutput;
            $fileName = $encoded_id . '_image.pdf'; // Adjust the file name as needed

            LogsHelper::log([
                'class_id' => $formData['id'],
                'class_type' => get_class(new VideoBrief()),
                'activity' => 'Video Brief Downloaded by '. $request->user()->email .'.',
                'type' => 'downloaded',
                'created_by' => Auth::user()->id,
                'code' => 200,
                'request' => json_encode(['encoded_id' => $encoded_id, 'fingerprint' => $fingerprint]),
                'response' => response()->json($formData)->content(),
            ], $this->logsDirectory);

            $this->briefActivityService->create($request, $formData['id'], 'video_brief', 'download', 'Brief Download Successfully', $fingerprint);

            $foundBrief = VideoBrief::where('encoded_id', $encoded_id)->first();
            if ($foundBrief) {
                $foundBrief->touch();
            }

            return response()->streamDownload(
                function () use ($imageData) {
                    echo $imageData;
                },
                $fileName,
                ['Content-Type' => 'application/pdf'] // Set the content type to PDF
            );
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }

    public function view(Request $request)
    {
        $isDisabled = true;
        if (Auth::check()) {
            $brief = $this->videoBriefService->fetchByEncodedID($request->id, true);
            $trackingApis = FileHelper::readFile($request->id);
        } else {
            $brief = $this->videoBriefService->fetchByEncodedID($request->id);
            $trackingApis = [];
        }


        return Inertia::render('VideoBriefView', [
            'isDisabled' => $isDisabled,
            'id' => $request->id,
            'brief' => $brief,
            'trackingApis' => $trackingApis,
        ]);
    }

    public function stringToArray($string)
    {
        $array = explode(",", $string);
        return $array;
    }

    public function briefDetail(Request $request){
        $brief = $this->videoBrief->with('briefLogs:id,id,brief_id,activity,type,created_by,request,response,pin,code,event,created_at', 'briefLink:id,link,brief_type,salesperson_id,status,visited,created_at', 'customer:id,first_name,last_name,email,phone,address')->findOrFail($request->id);

        $relatedBrief = [];
        $succeededPayments = collect();

        if($brief->customer){
            $relatedBrief = BriefHelper::getRelatedBriefs($brief->customer->id, $request->id, 'video');

            $customer = User::where('id', '=', $brief->customer->id)->with([
                'payments' => function ($query) {
                    $query->select('id', 'customer_id', 'payment_link_id', 'price', 'currency', 'paymentMethod', 'intent_id', 'charge_id', 'last_four', 'status', 'created_at');
                    $query->with([
                        'link' => function ($linkQ) {
                            $linkQ->select('id', 'customer_id', 'token', 'valid_till', 'item_name', 'price', 'currency', 'payment_gateway', 'status');
                            $linkQ->with('currencyCountry:id,aplha_code3,code,symbol');
                        }
                    ]);
                    $query->latest('id')->get();
                }
            ])->firstOrFail();

            $customer?->payments->each(function ($payment) use ($succeededPayments) {
                if ($payment->link->status == 2) { // Assuming '2' is the status for successful payments
                    $succeededPayments->push($payment->link);
                }
            });
        }

        $briefDetails = [
            'project_details' => [
                'project_name' => $brief->project_name,
                'name' => $brief->name,
                'contact' => $brief->phone,
                'email' => $brief->email,
                'industry' => $brief->industry,
            ],
            'project_requirements' => [
                'website' => $brief->website,
                'social_media' => $brief->social_media,
                'business_services' => $brief->business_services,
                'audience' => $brief->audience,
                'competitor' => $brief->competitor,
                'u_s_p' => $brief->u_s_p,
                'video_purpose' => $brief->video_purpose,
                'video_duration' => $brief->video_duration,
                'reference' => $brief->reference,
                'style' => $brief->style,
                'color' => $brief->color,
                'voice' => $brief->voice,
                'artist_accent' => $brief->artist_accent,
                'animation_style' => $brief->animation_style,
                'reference_video' => $brief->reference_video,
                'deadline' => $brief->deadline,
                'additional_comments' => $brief->additional_comments,
                'comments' => $brief->comments,
            ],
        ];

        $briefLogs = LogsHelper::getAllLogs($request->id, $this->logsDirectory);
        $briefLinkLogs = LogsHelper::getAllLogs($brief->briefLink->id, $this->linkLongDirectory);

        $formSubmissionAccuracy = 100;

        $errorLogsCount = 0;
        foreach ($briefLinkLogs['logs'] as $log) {
            if ($log['type'] === 'link.error') {
                $errorLogsCount++;
            }
        }

        $formSubmissionAccuracy -= $errorLogsCount;
        $formSubmissionAccuracy = max($formSubmissionAccuracy, 0);

        $briefActivities = VisitedBrief::where('brief_type', 'video_brief')->get();

        $activities = [];

        foreach ($briefActivities as $activity) {
            $data = json_decode($activity->data);

            $activities[] = [
                'id' => $activity['id'],
                'ip' => $activity['ip'],
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'userName' => isset($data->userAgent->name) ? $data->userAgent->name : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'createdAt' => $activity['created_at'],
            ];
        }

        $metaData = $brief->briefData()->select('key', 'value')->get();

        $page = [
            'title' => 'Video Brief Detail',
            'heading' => 'Video Brief Details',
            'route' => 'video.brief-detail.detail',
            'briefDirectory' => $this->logsDirectory,
            'metaDataRoute' => 'video.brief.updateMetadata',
            'deleteRoute' => 'video.brief.delete',
            'formRoute' => 'frontend.video.brief.view',
            'activityRoute' => 'video.brief-detail.activity',
            'tabs' => [
                [
                    'title' => 'Brief Insight',
                    'route' => 'video.brief-detail.detail',
                    'isActive' => true,
                ],
                // [
                //     'title' => 'Activity',
                //     'route' => 'video.brief-detail.activity',
                //     'isActive' => false,
                // ],
                // [
                //     'title' => 'Detail',
                //     'route' => 'video.brief-detail.details',
                //     'isActive' => false,
                // ]
            ],
        ];


        $briefActivitiesLogs = $this->briefActivityService->getBriefActivity($request->id, 'video_brief');

        $activitiesLogs = [];

        foreach ($briefActivitiesLogs as $activityLog) {
            $data = json_decode($activityLog->data);
            $loginHistory = $activityLog->loginHistory;

            if (isset($loginHistory->user)) {
                $firstName = $loginHistory->user->first_name ?? '';
                $lastName = $loginHistory->user->last_name ?? '';
                $fullName = trim($firstName . ' ' . $lastName);
            } else {
                $fullName = $activityLog['custom_id'];
            }

            $activitiesLogs[] = [
                'id' => $activityLog['id'],
                'custom_id' => $fullName,
                'session_id' => $activityLog['session_id'],
                'brief_id' => $activityLog['brief_id'],
                'brief_type' => $activityLog['brief_type'],
                'ip' => isset($data->ipInfo->ip) ? $data->ipInfo->ip : '',
                'country' => isset($data->ipInfo->country) ? $data->ipInfo->country : '',
                'os' => isset($data->userAgent->os) ? $data->userAgent->os : '',
                'activity' => $activityLog['activity'],
                'activity_desc' => $activityLog['activity_desc'],
                'fingerprint' => $activityLog['fingerprint'],
                'no_of_visits' => $activityLog['no_of_visits'],
                'created_at' => $activityLog['created_at'],
            ];
        }

        $authUser = Auth::user();
        $authUserRole = $authUser->roles->first()->name;

        // $this->briefActivityService->createViewActivity($request, $request->id, 'video_brief', 'view', 'Brief Detail Page Visited');

        return Inertia::render('Admin/Brief/Brief/BriefDetail', [
            'briefDetails' => $briefDetails,
            'brief' => $brief,
            'relatedBrief' => $relatedBrief,
            'metaData' => $metaData,
            'page' => $page,
            'activities' => $activities,
            'logs' => $briefLogs,
            'linkLogs' => $briefLinkLogs,
            'briefID' => $brief->id,
            'activitiesLogs' => $activitiesLogs,
            'authUserRole' => $authUserRole,
            'total_cost' => $succeededPayments->sum('price'),
            'submissionAccuracy' => $formSubmissionAccuracy,
        ]);
    }

    public function briefActivity(Request $request, $id)
    {
        $page = [
            'title' => 'Video Brief Activity',
            'heading' => 'Video Brief Activities',
            'route' => 'video.brief-detail.activity',
        ];

        // $page = [
        //     'title' => 'Video Brief Activity',
        //     'heading' => 'Video Brief Activities',
        //     'route' => 'video.brief-detail.activity',
        //     'tabs' => [
        //         [
        //             'title' => 'Brief Insight',
        //             'route' => 'video.brief-detail.detail',
        //             'isActive' => false,
        //         ],
        //         [
        //             'title' => 'Activity',
        //             'route' => 'video.brief-detail.activity',
        //             'isActive' => true,
        //         ],
        //         [
        //             'title' => 'Detail',
        //             'route' => 'video.brief-detail.details',
        //             'isActive' => false,
        //         ]
        //     ],
        // ];

        $columns = [
            ["key" => "name", "component" => "NameColumn", "text" => "VISITOR ID", "visible" => true, "fixed" => true],
            ["key" => "ip_address", "component" => "IPAddressColumn", "text" => "IP Address", "visible" => true],
            ["key" => "country", "component" => "LocationColumn", "text" => "Location", "visible" => true],
            ["key" => "os", "component" => "OsColumn", "text" => "OS", "visible" => true],
            ["key" => "activity", "component" => "ActivityColumn", "text" => "Activity", "visible" => true],
            ["key" => "fingerprint", "component" => "FingerprintColumn", "text" => "Fingerprint", "visible" => true],
            ["key" => "no_of_visits", "component" => "CountColumn", "text" => "VISITOR COUNT", "visible" => true],
            ["key" => "created_at", "component" => "CreatedAtColumn", "text" => "Created", "visible" => true],
        ];

        $activities = BriefActivity::select('id', 'custom_id', 'session_id', 'brief_id', 'brief_type', 'data', 'fingerprint', 'activity', 'activity_desc', 'no_of_visits', 'created_at')
            ->with('loginHistory.user')
            ->where('brief_id', $id)
            ->where('brief_type', 'video_brief')
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $nameParts = array_map('trim', explode(' ', $request->name, 2));
                $firstName = $nameParts[0] ?? null;
                $lastName = $nameParts[1] ?? null;

                $query->whereHas('loginHistory.user', function ($q) use ($firstName, $lastName) {
                    $q->where(function ($q) use ($firstName, $lastName) {
                        if ($firstName && $lastName) {
                            $q->where(function ($q) use ($firstName, $lastName) {
                                $q->where('first_name', 'like', '%' . $firstName . '%')
                                  ->where('last_name', 'like', '%' . $lastName . '%');
                            });
                        }
                        elseif ($firstName) {
                            $q->where('first_name', 'like', '%' . $firstName . '%')
                              ->orWhere('last_name', 'like', '%' . $lastName . '%');
                        }
                    });
                });
            })
            ->when((!empty($request->activitytype)), function ($query) use ($request) {
                $query->where('activity', $request->activitytype);
             })
            ->latest('id')
            ->paginate(30)
            ->withQueryString();


        return Inertia::render('Admin/Brief/Brief/BriefActivityDetail', [
            'page' => $page,
            'activities' => $activities,
            'columns' => $columns,
            'briefID' => $id,
        ]);
    }

    // public function details(Request $request){
    //     $brief = $this->videoBriefService->getBrief($request->id);

    //     $page = [
    //         'title' => 'Video Brief Detail',
    //         'heading' => 'Video Brief Details',
    //         'route' => 'video.brief-detail.detail',
    //         'formRoute' => 'frontend.video.brief.view',
    //         'tabs' => [
    //             [
    //                 'title' => 'Brief Insight',
    //                 'route' => 'video.brief-detail.detail',
    //                 'isActive' => false,
    //             ],
    //             // [
    //             //     'title' => 'Activity',
    //             //     'route' => 'video.brief-detail.activity',
    //             //     'isActive' => false,
    //             // ],
    //             // [
    //             //     'title' => 'Detail',
    //             //     'route' => 'video.brief-detail.details',
    //             //     'isActive' => true,
    //             // ]
    //         ],
    //     ];

    //     $briefDetails = [
    //         'first_column' => [
    //             'client_information' => [
    //                 'email' => $brief->briefLink->customer->email,
    //                 'client_name' => $brief->briefLink->customer->first_name . ' ' . $brief->briefLink->customer->last_name,
    //                 'contact_number' => $brief->briefLink->customer->phone,
    //             ],
    //         ],
    //         'second_column' => [
    //             'project_information' => [
    //                 'name_of_the_project' => $brief->project_name,
    //                 'business_industry' => $brief->industry,
    //                 'current_website_address_(if applicable)' => $brief->website,
    //                 'current_social_media_profile_(if applicable)' => $brief->social_media,
    //                 'business_description' => $brief->business_description,
    //                 'business_products_/_services' => $brief->business_services,
    //                 'target_audience_or_customers_you_want_to_target' => $brief->audience,
    //                 'competitors_or_similar_businesses_(for reference purpose)' => $brief->competitor,
    //                 'how_is_your_company_different_from_your_competitors?_(your_unique_selling_proposition)' => $brief->u_s_p,
    //                 'additional_comments_(e.g.,_my_vision_for_the_script)' => $brief->comments,
    //                 'purpose_of_video' => $brief->video_purpose,
    //                 'what_is_the_duration_of_the_video' => $brief->video_duration,
    //                 'would_you_like_to_follow_any_reference_video_(e.g.,_character_animation, motion_graphics,_or_whiteboard)' => $brief->reference,
    //                 'would_you_like_to_follow_any_specific_style_of_characters?' => $brief->style,
    //                 'would_you_like_to_use_any_specific_color_palettes?' => $brief->color,
    //                 'voice-over_preference' => explode(',', $brief->voice),
    //                 'voice-over_artist_accent' => $brief->artist_accent,
    //                 'animation_style' => $brief->animation_style,
    //                 'any_reference_video_you\'d_like_us_to_use_as_inspiration?' => $brief->reference_video,
    //                 'are_you_working_with_any_deadline' => $brief->deadline,
    //                 'additional_comments_over_the_video' => $brief->additional_comments,
    //             ],
    //         ]
    //     ];

    //     return Inertia::render('Admin/Brief/Brief/BriefDetails', [
    //         'brief' => $brief,
    //         'briefDetails' => $briefDetails,
    //         'page' => $page,
    //         'briefID' => $brief->id,
    //     ]);
    // }

    public function updateMetadata(Request $request){
        try {
            $query = $this->videoBrief->findOrFail($request->id);
            BriefHelper::updateMetadata($query, $request->metadata);

            $foundBrief = VideoBrief::findOrFail($request->id);
            if ($foundBrief) {
                $foundBrief->touch();
            }

            $this->briefActivityService->create($request, $request->id, 'video_brief', 'update', 'Metadata Updated Successfully', $request->fingerprint);

        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update metadata'], 500);
        }
    }

}
