<?php

namespace App\Http\Controllers\Admin\Brief;

use App\Helper\Helper\NotificationHelper;
use App\Helper\LogsHelper;
use App\Http\Controllers\Controller;
use App\Models\BriefLink;
use App\Models\BriefLinkHistory;
use App\Models\BriefLinkVisitor;
use App\Models\User;
use App\Models\VisitedBrief;
use App\Services\ApiService;
use App\Services\BriefLinkHistoryService;
use App\Services\GenerateUniqueID;
use App\Services\MetadataService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class BriefLinkController extends Controller
{
    private $briefLink, $briefLinkObj, $apiService, $generateUniqueID, $briefLinkHistoryService;
    private $linkLongDirectory = 'brief_link';

    public function __construct(BriefLink $briefLink, ApiService $apiService, GenerateUniqueID $generateUniqueID, BriefLinkHistoryService $briefLinkHistoryService){
        $this->briefLink = $briefLink;
        $this->apiService = $apiService;
        $this->briefLinkObj = new $briefLink;
        $this->generateUniqueID = $generateUniqueID;
        $this->briefLinkHistoryService = $briefLinkHistoryService;
    }

    public function index(Request $request){

        $briefLinks = BriefLink::select('id', 'customer_id', 'link', 'brief_type', 'salesperson_id', 'status', 'visited', 'created_at')
            ->with(['salesperson:id,first_name,last_name', 'customer:id,first_name,last_name,phone,email,company'])
            ->when(!empty($request->dateSearchBy), function ($query) use ($request) {
                $parsedDate = null;
                if ($request->timezone == 'eastern' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('America/New_York');
                }

                if ($request->timezone == 'utc-5' && $request->dateSearchBy != '>') {
                    $parsedDate = Carbon::parse($request->date)->setTimezone('UTC')->addHours(5);
                }
                switch ($request->dateSearchBy) {
                    case '>':
                        if (!empty($request->date)) {
                            $date = date('Y-m-d', strtotime('-' . $request->date . ' ' . $request->durationType));
                            $query->whereDate('created_at', '>=', $date);
                        }
                        break;
                    case '=':
                        if (!empty($request->date)) {
                            $query->whereDate('created_at', $parsedDate);
                        }
                        break;
                    case '<>':
                        if (!empty($request->dateBetween['start']) && !empty($request->dateBetween['end'])) {
                            $startDate = Carbon::parse($request->dateBetween['start']);
                            $endDate = Carbon::parse($request->dateBetween['end']);

                            // Adjust time if provided
                            if (!empty($request->timeBetween['start'])) {
                                $startDate->setTimeFromTimeString($request->timeBetween['start'].":00");
                            }

                            if (!empty($request->timeBetween['end'])) {
                                $endDate->setTimeFromTimeString($request->timeBetween['end'].":59");
                            }

                            // Adjust timezone
                            if ($request->timezone == 'eastern') {
                                $startDate = Carbon::parse($startDate)->setTimezone('America/New_York');
                                $endDate = Carbon::parse($endDate)->setTimezone('America/New_York');
                            }

                            // Adjust timezone
                            if ($request->timezone == 'utc-5') {
                                $startDate->setTimezone('UTC')->addHours(5);
                                $endDate->setTimezone('UTC')->addHours(5);
                            }

                            // Apply date and time range filter
                            $query->whereBetween('created_at', [$startDate->toDateTimeString(), $endDate->toDateTimeString()]);
                        }
                        break;
                    case '>=':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time);
                                $query->where('created_at', '>=', $dateTime);
                            } else {
                                $query->whereDate('created_at', '>=', $parsedDate);
                            }
                        }
                        break;
                    case '<':
                        if (!empty($request->date)) {
                            if (!empty($request->time)) {
                                $dateTime = Carbon::parse($request->date . ' ' . $request->time . ":59");
                                $query->where('created_at', '<', $dateTime);
                            } else {
                                $query->whereDate('created_at', '<', $parsedDate);
                            }
                        }
                        break;
                }
            })
            ->when(!empty($request->name), function ($query) use ($request) {
                $query->whereHas('customer', function ($q) use ($request) {
                    $q->where('first_name', 'like', '%' . $request->name . '%')
                    ->orWhere('last_name', 'like', '%' . $request->name . '%');
                });
            })
            ->when(!empty($request->customerEmail), function ($query) use ($request) {
                $query->whereHas('customer', function ($q) use ($request) {
                    $q->where('email', $request->customerEmail);
                });
            })
            ->when(!empty($request->phone), function ($query) use ($request) {
                $query->whereHas('customer', function ($q) use ($request) {
                    $q->where('phone', $request->phone);
                });
            })
            ->when(!empty($request->company), function ($query) use ($request) {
                $query->whereHas('customer', function ($q) use ($request) {
                    $q->where('company', $request->company);
                });
            })
            ->when((!empty($request->type)), function ($query) use ($request) {
                $query->where('brief_type', $request->type.'_brief');
             })
            ->when((!empty($request->status)), function ($query) use ($request) {
               $query->where('status', $request->status);
            })
            ->latest('created_at')
            ->paginate(20)
            ->withQueryString();

        $columns = [
            ["key" => "link_copy", "component" => "LinkCopyColumn", "text" => "Link Copy", "visible" => true, "fixed" => true],
            ["key" => "brief_type", "component" => "BriefTypeColumn", "text" => "Brief Type", "visible" => true],
            ["key" => "customer.first_name customer.last_name", "component" => "NameColumn", "text" => "Customer Name", "visible" => false],
            ["key" => "customer.phone", "component" => "PhoneColumn", "text" => "Phone", "visible" => false],
            ["key" => "customer.email", "component" => "EmailColumn", "text" => "Email", "visible" => true],
            ['key' => "salesperson.first_name", 'component' => 'SalesPersonColumn', 'text' => 'Sales Person', 'visible' => true],
            ["key" => "customer.company", "component" => "CompanyColumn", "text" => "Company Name", "visible" => false],
            ["key" => "status", "component" => "StatusColumn", "text" => "Status", "visible" => true],
            ["key" => "visited", "component" => "VisitedColumn", "text" => "Visited", "visible" => true],
            ['key' => 'created_at', 'component' => 'CreatedAtColumn', 'text' => 'Date', 'visible' => true],
        ];

        return Inertia::render('Admin/Brief/Brief/LinkList',[
            'columns' => $columns,
            'briefLinks' => $briefLinks
        ]);
    }

    public function detail(Request $request)
    {
        $briefLink =  BriefLink::with([
            'customer',
            'salesperson',
            // 'briefLinkLog' => function ($logs) {
            //     $logs->latest('created_at');
            // }
        ])->findOrFail($request->id);

        $briefLinkHistory = BriefLinkHistory::query()
                    ->where('brief_link_id', $request->id)
                    ->latest()
                    ->get();

        $logs = LogsHelper::getAllLogs($request->id,  $this->linkLongDirectory);

        return  Inertia::render('Admin/Brief/Brief/LinkDetail', [
            'briefLink' => $briefLink,
            'logs' => $logs,
            'briefLinkHistory' => $briefLinkHistory,
        ]);
    }

    public function form(Request $request)
    {

        $validator = Validator::make(["token" => $request->token], [
            'token' => 'required|string|exists:brief_links,link',
        ]);

        if ($validator->fails()) {
            return redirect()->route('brief.link.list')
                ->withErrors($validator)
                ->withInput();
        }

        $foundBrief = $this->briefLink->where('link', $request->token)->first();

        if(!empty($foundBrief) && $foundBrief->status == "not_submitted"){
            $customer = User::find($foundBrief->customer_id);

            $briefType = $foundBrief->brief_type;
            $briefLinkID = $foundBrief->id;

            $isDisabled = false;
            if ($request->id) {
                $isDisabled = true;
            }

            $submitRoute = "";

            if($briefType == "logo_brief"){
                $submitRoute = 'brief.submit';
            }else if($briefType == "web_brief"){
                $submitRoute = "web.brief.submit";
            }else if($briefType == "video_brief"){
                $submitRoute = "video.brief.submit";
            }else if($briefType == "smm_brief"){
                $submitRoute = "smm.brief.submit";
            }else if($briefType == "mobile_app_brief"){
                $submitRoute = "app.brief.submit";
            }else if($briefType == "sem_brief"){
                $submitRoute = "sem.brief.submit";
            }

            if ($request->hasHeader('X-Forwarded-For')) {
                $clientIP = $request->header('X-Forwarded-For');
            } else {
                $clientIP = $request->ip();
            }

            $recordExists = VisitedBrief::select('id')->where('ip', $clientIP)
                ->where('created_at', '>=', Carbon::now()->subMinutes(5))
                ->exists();

            $data = $this->apiService->fetchInfoAndUserAgent($request);

            if (isset($data['ipInfo']['country-code']) && $data['ipInfo']['country-code'] == 'US') {
                $foundBrief->visited = "visited";
                $foundBrief->save();
            }

            if (!$recordExists) {
                VisitedBrief::create([
                    'ip' => $clientIP,
                    'brief_type' => $briefType,
                    'data' => response()->json($data)->content()
                ]);
            }

            return Inertia::render('BriefForm', [
                'isDisabled' => $isDisabled,
                'id' => $request->id,
                'briefType' => $briefType,
                'briefLinkID' => $briefLinkID,
                'customer' => $customer,
                'submitRoute' => $submitRoute
            ]);

        } else {
            return redirect()->route('brief.link.expired')->with('success', 'Brief expired');
        }
    }

    public function add(Request $request)
    {
        $page = [
            'title' => 'Brief Link Generator'
        ];

        $briefTypes = $this->briefLinkObj->getBriefTypes();
        $hideFields = true;
        $customer = User::find($request->customer_id);

        return Inertia::render('Admin/Brief/Brief/LinkAddEdit', [
            'page' => $page,
            'briefTypes' => $briefTypes,
            'hideFields' => $hideFields,
            'customer' => $customer,
            'method' => 'add',
        ]);
    }

    public function edit(Request $request)
    {
        $briefLink = $this->briefLink->whereId($request->id)->with('customer')->first();

        $page = [
            'title' => 'Brief Link Edit'
        ];

        $briefTypes = $this->briefLinkObj->getBriefTypes();
        $hideFields = true;
        return Inertia::render('Admin/Brief/Brief/LinkAddEdit', [
            'page' => $page,
            'briefTypes' => $briefTypes,
            'hideFields' => $hideFields,
            'briefLink' => $briefLink,
            'method' => 'edit',
        ]);
    }

    public function create(Request $request)
    {
        try {
            $rules =  [
                'briefType' => 'required',
                'existing_customer' => 'required',
                'cid' => 'nullable|required_if:existing_customer,yes',
            ];

            $CustomMessages = [
                'briefType.required' => 'Brief Type is required.',
                'existing_customer.required' => 'This field is required.',
                'cid.required_if' => 'Customer Name is required',
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if ($validator->fails()) {
                return Redirect::route('brief.link.add')
                    ->withErrors($validator)
                    ->withInput();
            }

            $briefLink = new $this->briefLink;

            $prefix='brief_link';
            $length = 8;
            $uniqueBriefId = $this->generateUniqueID->create($prefix, $length);
            $briefLink->custom_id = $uniqueBriefId;

            $briefLink->customer_id =  ($request->existing_customer == 'yes' ) ? $request->cid : null;
            $briefLink->existing_customer = $request->existing_customer;
            $briefLink->link = sha1(uniqid(Carbon::now()->toDateTimeString() . $request->user()->id . $request->cid, true));
            $briefLink->brief_type = $request->briefType;
            $briefLink->salesperson_id = $request->user()->id;
            $briefLink->status = 'not_submitted';
            $briefLink->visited = 'not_visited';

            $briefLink->save();

            $this->briefLinkHistoryService->create($briefLink->id, $request->ip(), auth()->id(), $request->header('User-Agent'), "Create");

            LogsHelper::log([
                'class_type' => 'brief_link',
                'class_id' => $briefLink->id,
                'activity' => 'Brief Link Created by ' . $request->user()->email . '.',
                'type' => 'link.created',
                'request' => $request,
                'code' => 200,
                'response' => response()->json($briefLink)->content(),
                'created_by' => $request->user()->id,
            ], $this->linkLongDirectory);

            NotificationHelper::notify('brief_link', 'Brief Link Created ', [
                'id' => $briefLink->id,
                'request' => response()->json($request->all())->content(),
                'response' => response()->json($briefLink)->content(),
                'message' => 'Brief Link Created Successful',
            ], Auth::user()->id);

            Session::flash('success', true);
            Session::flash('message', 'New Brief link generated.');


            return Redirect::route('brief.link.list');
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());

            return Redirect::route('brief.link.add');
        }

    }

    public function updateMetadata(Request $request)
    {
        $briefLink = $this->briefLink->findOrFail($request->id);

        $metadataService = new MetadataService($briefLink, 'data');
        $metadataService->update($request->metadata);
    }

    public function delete(Request $request)
    {
        $briefLink = $this->briefLink->findOrFail($request->id);
        $briefLink->delete();

        $this->briefLinkHistoryService->create($briefLink->id, $request->ip(), auth()->id(), $request->header('User-Agent'), "Deactivate");

        return redirect()->route('brief.link.list');
    }

    public function expired()
    {
        $pageProperties = [
            'heading' => 'Link Expired',
            'description' => ['Brief Link has been Expired.']
        ];

        return Inertia::render('BriefExpired', [
            'pageProperties' => $pageProperties,
        ]);
    }

    public function update(Request $request, $id)
    {
        try {
            $rules =  [
                'briefType' => 'required',
                'existing_customer' => 'required',
                'cid' => 'nullable|required_if:existing_customer,yes',
            ];

            $CustomMessages = [
                'briefType.required' => 'Brief Type is required.',
                'cid.required_if' => 'Customer Name is required',
            ];

            $validator = Validator::make($request->all(), $rules, $CustomMessages);

            if ($validator->fails()) {
                return Redirect::route('brief.link.edit.id', ['id' => $id])
                    ->withErrors($validator)
                    ->withInput();
            }

            $briefLink = $this->briefLink->findOrFail($id);

            $briefLink->customer_id = ($request->existing_customer == 'yes' ) ? $request->cid : null;
            $briefLink->existing_customer = $request->existing_customer;
            $briefLink->brief_type = $request->briefType;
            $briefLink->save();

            $this->briefLinkHistoryService->create($briefLink->id, $request->ip(), auth()->id(), $request->header('User-Agent'), "Update");

            LogsHelper::log([
                'class_type' => 'brief_link',
                'class_id' => $briefLink->id,
                'activity' => 'Brief Link Updated by ' . $request->user()->email . '.',
                'type' => 'link.updated',
                'request' => $request,
                'code' => 200,
                'response' => response()->json($briefLink)->content(),
                'created_by' => $request->user()->id,
            ], $this->linkLongDirectory);

            NotificationHelper::notify('brief_link', 'Brief Link Updated ', [
                'id' => $briefLink->id,
                'request' => response()->json($request->all())->content(),
                'response' => response()->json($briefLink)->content(),
                'message' => 'Brief Link Created Successful',
            ], Auth::user()->id);

            Session::flash('success', true);
            Session::flash('message', 'Brief link Updated.');


            return Redirect::route('brief.link.list');
        } catch (Exception $e) {
            Session::flash('success', false);
            Session::flash('message', $e->getMessage());

            return Redirect::route('brief.link.add');
        }

    }

    public function statusChange(Request $request)
    {
        foreach ($request->ids as $id) {
            $briefLink = $this->briefLink->find($id);

            if ($request->status == "deactivate") {
                $briefLink->status = "deactivate";
            } else if ($request->status == "not_submitted") {
                $briefLink->status = "not_submitted";
            }

            $briefLink->save();
        }
    }

    public function createLog(Request $request){

        LogsHelper::log([
            'class_type' => 'brief_link',
            'class_id' => $request->briefLinkId,
            'activity' => $request->activity,
            'type' => 'link.error',
            'request' => $request,
            'code' => $request->statusCode,
            'response' => response()->json($request->response)->content(),
            'created_by' => 1,
        ], $this->linkLongDirectory);

    }

}
